"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, /**
 * CopyFactory client for strategy signal requests
 */ "default", {
    enumerable: true,
    get: function() {
        return StrategySignalClient;
    }
});
const _randomstring = /*#__PURE__*/ _interop_require_default(require("randomstring"));
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
let StrategySignalClient = class StrategySignalClient {
    /**
   * Constructs CopyFactory strategy signal client instance
   * @param {string} accountId strategy provider id
   * @param {string} strategyId strategy id
   * @param {Object} host host data
   * @param {DomainClient} domainClient domain client
   */ constructor(accountId, strategyId, host, domainClient){
        this._accountId = accountId;
        this._strategyId = strategyId;
        this._domainClient = domainClient;
        this._host = host;
    }
    /**
   * Generates random signal id
   * @return {String} signal id
   */ generateSignalId() {
        return _randomstring.default.generate(8);
    }
    /**
   * CopyFactory external signal update payload
   * @typedef {Object} CopyFactoryExternalSignalUpdate
   * @property {String} symbol trade symbol
   * @property {String} type trade type (one of POSITION_TYPE_BUY, POSITION_TYPE_SELL, ORDER_TYPE_BUY_LIMIT, ORDER_TYPE_SELL_LIMIT, 
   * ORDER_TYPE_BUY_STOP, ORDER_TYPE_SELL_STOP)
   * @property {Date} time time the signal was emitted at
   * @property {Date} [updateTime] last time of the signal update
   * @property {Number} volume volume traded
   * @property {Number} [magic] expert advisor id
   * @property {Number} [stopLoss] stop loss price
   * @property {Number} [takeProfit] take profit price
   * @property {Number} [openPrice] pending order open price
   */ /**
   * CopyFactory external signal
   * @typedef {CopyFactoryExternalSignalUpdate} CopyFactoryExternalSignal
   * @property {String} id external signal id
   */ /**
   * CopyFactory external signal remove payload
   * @typedef {Object} CopyFactoryExternalSignalRemove
   * @property {Date} time the time signal was removed (closed) at
   */ /**
   * Returns active external signals of a strategy. Requires access to
   * copyfactory-api:rest:public:external-signals:getSignals method which is included into reader role.
   * Requires access to strategy, account resources.
   * @param {String} strategyId strategy id
   * @returns {Promise<Array<CopyFactoryExternalSignal>>}
   */ getExternalSignals() {
        const opts = {
            url: `/users/current/strategies/${this._strategyId}/external-signals`,
            method: "GET",
            headers: {
                "auth-token": this._domainClient.token
            },
            json: true
        };
        return this._domainClient.requestSignal(opts, this._host, this._accountId);
    }
    /**
   * Updates external signal for a strategy. See
   * https://metaapi.cloud/docs/copyfactory/restApi/api/trading/updateExternalSignal/
   * @param {String} signalId external signal id (should be 8 alphanumerical symbols)
   * @param {CopyFactoryExternalSignalUpdate} signal signal update payload
   * @return {Promise} promise which resolves when the external signal is updated
   */ updateExternalSignal(signalId, signal) {
        const opts = {
            url: `/users/current/strategies/${this._strategyId}/external-signals/${signalId}`,
            method: "PUT",
            headers: {
                "auth-token": this._domainClient.token
            },
            data: signal,
            json: true
        };
        return this._domainClient.requestSignal(opts, this._host, this._accountId);
    }
    /**
   * CopyFactory external signal remove payload
   * @typedef {Object} CopyFactoryExternalSignalRemove
   * @property {Date} time the time signal was removed (closed) at
   */ /**
   * Removes (closes) external signal for a strategy. See
   * https://metaapi.cloud/docs/copyfactory/restApi/api/trading/removeExternalSignal/
   * @param {String} signalId external signal id
   * @param {CopyFactoryExternalSignalRemove} signal signal removal payload
   * @return {Promise} promise which resolves when the external signal is removed
   */ removeExternalSignal(signalId, signal) {
        const opts = {
            url: `/users/current/strategies/${this._strategyId}/external-signals/${signalId}/remove`,
            method: "POST",
            headers: {
                "auth-token": this._domainClient.token
            },
            data: signal,
            json: true
        };
        return this._domainClient.requestSignal(opts, this._host, this._accountId);
    }
};

//# sourceMappingURL=data:application/json;base64,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