"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, /**
 * Stopout event listener manager
 */ "default", {
    enumerable: true,
    get: function() {
        return StopoutListenerManager;
    }
});
const _metaapiclient = /*#__PURE__*/ _interop_require_default(require("../../metaapi.client"));
const _randomstring = /*#__PURE__*/ _interop_require_default(require("randomstring"));
const _logger = /*#__PURE__*/ _interop_require_default(require("../../../logger"));
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
let StopoutListenerManager = class StopoutListenerManager extends _metaapiclient.default {
    /**
   * Constructs stopout listener manager instance
   * @param {DomainClient} domainClient domain client
   */ constructor(domainClient){
        super(domainClient);
        this._domainClient = domainClient;
        this._stopoutListeners = {};
        this._errorThrottleTime = 1000;
        this._logger = _logger.default.getLogger("StopoutListenerManager");
    }
    /**
   * Returns the dictionary of stopout listeners
   * @returns {Object} dictionary of stopout listeners
   */ get stopoutListeners() {
        return this._stopoutListeners;
    }
    /**
   * Adds a stopout listener
   * @param {StopoutListener} listener 
   * @param {String} [accountId] account id
   * @param {String} [strategyId] strategy id
   * @param {Number} [sequenceNumber] event sequence number
   * @returns {String} stopout listener id
   */ addStopoutListener(listener, accountId, strategyId, sequenceNumber) {
        const listenerId = _randomstring.default.generate(10);
        this._stopoutListeners[listenerId] = listener;
        this._startStopoutEventJob(listenerId, listener, accountId, strategyId, sequenceNumber);
        return listenerId;
    }
    /**
   * Removes stopout listener by id
   * @param {String} listenerId listener id 
   */ removeStopoutListener(listenerId) {
        delete this._stopoutListeners[listenerId];
    }
    async _startStopoutEventJob(listenerId, listener, accountId, strategyId, sequenceNumber) {
        let throttleTime = this._errorThrottleTime;
        while(this._stopoutListeners[listenerId]){
            const opts = {
                url: "/users/current/stopouts/stream",
                method: "GET",
                params: {
                    previousSequenceNumber: sequenceNumber,
                    subscriberId: accountId,
                    strategyId: strategyId,
                    limit: 1000
                },
                headers: {
                    "auth-token": this._token
                },
                json: true
            };
            try {
                const packets = await this._domainClient.requestCopyFactory(opts, true);
                await listener.onStopout(packets);
                throttleTime = this._errorThrottleTime;
                if (this._stopoutListeners[listenerId] && packets.length) {
                    sequenceNumber = packets.slice(-1)[0].sequenceNumber;
                }
            } catch (err) {
                await listener.onError(err);
                this._logger.error(`Failed to retrieve stopouts stream for strategy ${strategyId}, ` + `listener ${listenerId}, retrying in ${Math.floor(throttleTime / 1000)} seconds`, err);
                await new Promise((res)=>setTimeout(res, throttleTime));
                throttleTime = Math.min(throttleTime * 2, 30000);
            }
        }
    }
};

//# sourceMappingURL=data:application/json;base64,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