import MetaApiClient from '../metaapi.client';
import SubscriberSignalClient from './subscriberSignal.client';
import StrategySignalClient from './strategySignal.client';
import ConfigurationClient from './configuration.client';
import DomainClient from '../domain.client';
import StopoutListener from './streaming/stopoutListener';
import UserLogListener from './streaming/userLogListener';
import { CopyFactoryStrategyStopout, CopyFactoryStrategyStopoutReason, CopyFactoryUserLogMessage } from './trading.client.schemas';
export * from './trading.client.schemas';
/**
 * metaapi.cloud CopyFactory trading API (trade copying trading API) client (see
 * https://metaapi.cloud/docs/copyfactory/)
 */
export default class TradingClient extends MetaApiClient {
    private _configurationClient;
    private _stopoutListenerManager;
    private _userLogListenerManager;
    /**
     * Constructs CopyFactory trading API client instance
     * @param {DomainClient} domainClient domain client
     * @param {ConfigurationClient} configurationClient configuration client
     */
    constructor(domainClient: DomainClient, configurationClient: ConfigurationClient);
    /**
     * Resynchronizes the account. See
     * https://metaapi.cloud/docs/copyfactory/restApi/api/trading/resynchronize/
     * @param {string} accountId account id
     * @param {Array<string>} [strategyIds] array of strategy ids to recynchronize. Default is to synchronize all
     * strategies
     * @param {Array<string>} [positionIds] array of position ids to resynchronize. Default is to synchronize all
     * positions
     * @return {Promise} promise which resolves when resynchronization is scheduled
     */
    resynchronize(accountId: string, strategyIds?: Array<string>, positionIds?: Array<string>): Promise<any>;
    /**
     * Generates an instance of signal client for a subscriber
     * @param {string} subscriberId subscriber account id
     */
    getSubscriberSignalClient(subscriberId: string): Promise<SubscriberSignalClient>;
    /**
     * Generates an instance of signal client for a strategy
     * @param {string} strategyId strategy id
     */
    getStrategySignalClient(strategyId: string): Promise<StrategySignalClient>;
    /**
     * Returns subscriber account stopouts. See
     * https://metaapi.cloud/docs/copyfactory/restApi/api/trading/getStopOuts/
     * @param {string} subscriberId subscriber id
     * @return {Promise<Array<CopyFactoryStrategyStopout>>} promise which resolves with stopouts found
     */
    getStopouts(subscriberId: string): Promise<Array<CopyFactoryStrategyStopout>>;
    /**
     * Resets subscription stopouts. See
     * https://metaapi.cloud/docs/copyfactory/restApi/api/trading/resetSubscriptionStopOuts/
     * @param {string} subscriberId subscriber id
     * @param {string} strategyId strategy id
     * @param {CopyFactoryStrategyStopoutReason} reason stopout reason to reset
     * yearly-equity, monthly-equity, daily-equity
     * @return {Promise} promise which resolves when the stopouts are reset
     */
    resetSubscriptionStopouts(subscriberId: string, strategyId: string, reason: CopyFactoryStrategyStopoutReason): Promise<any>;
    /**
     * Resets subscriber stopouts. See
     * https://metaapi.cloud/docs/copyfactory/restApi/api/trading/resetSubscriberStopOuts/
     * @param {string} subscriberId subscriber id
     * @param {CopyFactoryStrategyStopoutReason} reason stopout reason to reset
     * yearly-equity, monthly-equity, daily-equity
     * @return {Promise} promise which resolves when the stopouts are reset
     */
    resetSubscriberStopouts(subscriberId: string, reason: CopyFactoryStrategyStopoutReason): Promise<any>;
    /**
     * Returns copy trading user log for an account and time range. See
     * https://metaapi.cloud/docs/copyfactory/restApi/api/trading/getUserLog/
     * @param {string} subscriberId subscriber id
     * @param {Date} [startTime] time to start loading data from
     * @param {Date} [endTime] time to stop loading data at
     * @param {string} [strategyId] strategy id filter
     * @param {string} [positionId] position id filter
     * @param {'DEBUG'|'INFO'|'WARN'|'ERROR'} [level] minimum severity level
     * @param {Number} [offset] pagination offset. Default is 0
     * @param {Number} [limit] pagination limit. Default is 1000
     * @return {Promise<Array<CopyFactoryUserLogMessage>>} promise which resolves with log records found
     */
    getUserLog(subscriberId: string, startTime?: Date, endTime?: Date, strategyId?: string, positionId?: string, level?: 'DEBUG' | 'INFO' | 'WARN' | 'ERROR', offset?: number, limit?: number): Promise<Array<CopyFactoryUserLogMessage>>;
    /**
     * Returns event log for CopyFactory strategy, sorted in reverse chronological order. See
     * https://metaapi.cloud/docs/copyfactory/restApi/api/trading/getStrategyLog/
     * @param {string} strategyId strategy id to retrieve log for
     * @param {Date} [startTime] time to start loading data from
     * @param {Date} [endTime] time to stop loading data at
     * @param {string} [positionId] position id filter
     * @param {'DEBUG'|'INFO'|'WARN'|'ERROR'} [level] minimum severity level
     * @param {Number} [offset] pagination offset. Default is 0
     * @param {Number} [limit] pagination limit. Default is 1000
     * @return {Promise<Array<CopyFactoryUserLogMessage>>} promise which resolves with log records found
     */
    getStrategyLog(strategyId: string, startTime?: Date, endTime?: Date, positionId?: string, level?: 'DEBUG' | 'INFO' | 'WARN' | 'ERROR', offset?: number, limit?: number): Promise<Array<CopyFactoryUserLogMessage>>;
    /**
     * Adds a stopout listener and creates a job to make requests
     * @param {StopoutListener} listener stopout listener
     * @param {string} [accountId] account id
     * @param {string} [strategyId] strategy id
     * @param {Number} [sequenceNumber] sequence number
     * @return {string} listener id
     */
    addStopoutListener(listener: StopoutListener, accountId?: string, strategyId?: string, sequenceNumber?: number): string;
    /**
     * Removes stopout listener and cancels the event stream
     * @param {string} listenerId stopout listener id
     */
    removeStopoutListener(listenerId: string): void;
    /**
     * Adds a strategy log listener and creates a job to make requests
     * @param {UserLogListener} listener user log listener
     * @param {string} strategyId strategy id
     * @param {Date} [startTime] log search start time
     * @param {string} [positionId] position id filter
     * @param {'DEBUG'|'INFO'|'WARN'|'ERROR'} [level] minimum severity level
     * @param {Number} [limit] log pagination limit
     * @return {string} listener id
     */
    addStrategyLogListener(listener: UserLogListener, strategyId: string, startTime?: Date, positionId?: string, level?: 'DEBUG' | 'INFO' | 'WARN' | 'ERROR', limit?: number): string;
    /**
     * Removes strategy log listener and cancels the event stream
     * @param {string} listenerId strategy log listener id
     */
    removeStrategyLogListener(listenerId: string): void;
    /**
     * Adds a subscriber log listener and creates a job to make requests
     * @param {UserLogListener} listener user log listener
     * @param {string} subscriberId subscriber id
     * @param {Date} [startTime] log search start time
     * @param {string} [strategyId] strategy id filter
     * @param {string} [positionId] position id filter
     * @param {'DEBUG'|'INFO'|'WARN'|'ERROR'} [level] minimum severity level
     * @param {Number} [limit] log pagination limit
     * @return {string} listener id
     */
    addSubscriberLogListener(listener: UserLogListener, subscriberId: string, startTime?: Date, strategyId?: string, positionId?: string, level?: 'DEBUG' | 'INFO' | 'WARN' | 'ERROR', limit?: number): string;
    /**
     * Removes subscriber log listener and cancels the event stream
     * @param {string} listenerId subscriber log listener id
     */
    removeSubscriberLogListener(listenerId: string): void;
}
