"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "default", {
    enumerable: true,
    get: function() {
        return ConnectionRegistry;
    }
});
const _streamingMetaApiConnection = /*#__PURE__*/ _interop_require_default(require("./streamingMetaApiConnection"));
const _rpcMetaApiConnection = /*#__PURE__*/ _interop_require_default(require("./rpcMetaApiConnection"));
const _streamingMetaApiConnectionInstance = /*#__PURE__*/ _interop_require_default(require("./streamingMetaApiConnectionInstance"));
const _rpcMetaApiConnectionInstance = /*#__PURE__*/ _interop_require_default(require("./rpcMetaApiConnectionInstance"));
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
let ConnectionRegistry = class ConnectionRegistry {
    /**
   * Creates and returns a new account streaming connection if doesnt exist, otherwise returns old
   * @param {MetatraderAccount} account MetaTrader account id to connect to
   * @param {HistoryStorage} historyStorage terminal history storage
   * @param {Date} [historyStartTime] history start time
   * @return {StreamingMetaApiConnection} streaming metaapi connection
   */ connectStreaming(account, historyStorage, historyStartTime) {
        if (!this._streamingConnections[account.id]) {
            this._streamingConnections[account.id] = new _streamingMetaApiConnection.default(this._options, this._metaApiWebsocketClient, this._terminalHashManager, account, historyStorage, this, historyStartTime, this._refreshSubscriptionsOpts);
        }
        const instance = new _streamingMetaApiConnectionInstance.default(this._metaApiWebsocketClient, this._streamingConnections[account.id]);
        this._streamingConnectionInstances[account.id] = this._streamingConnectionInstances[account.id] || [];
        this._streamingConnectionInstances[account.id].push(instance);
        return instance;
    }
    /**
   * Removes a streaming connection from registry
   * @param {MetatraderAccount} account MetaTrader account to remove from registry
   */ async removeStreaming(account) {
        if (this._streamingConnections[account.id]) {
            delete this._streamingConnections[account.id];
            delete this._streamingConnectionInstances[account.id];
        }
        if (!this._rpcConnections[account.id]) {
            await this._closeLastConnection(account);
        }
    }
    /**
   * Creates and returns a new account rpc connection if doesnt exist, otherwise returns old
   * @param {MetatraderAccount} account MetaTrader account id to connect to
   * @returns {RpcMetaApiConnection} rpc metaapi connection
   */ connectRpc(account) {
        if (!this._rpcConnections[account.id]) {
            this._rpcConnections[account.id] = new _rpcMetaApiConnection.default(this._options, this._metaApiWebsocketClient, account, this);
        }
        const instance = new _rpcMetaApiConnectionInstance.default(this._metaApiWebsocketClient, this._rpcConnections[account.id]);
        this._rpcConnectionInstances[account.id] = this._rpcConnectionInstances[account.id] || [];
        this._rpcConnectionInstances[account.id].push(instance);
        return instance;
    }
    /**
   * Removes an RPC connection from registry
   * @param {MetatraderAccount} account MetaTrader account to remove from registry
   */ async removeRpc(account) {
        if (this._rpcConnections[account.id]) {
            delete this._rpcConnections[account.id];
            delete this._rpcConnectionInstances[account.id];
        }
        if (!this._streamingConnections[account.id]) {
            await this._closeLastConnection(account);
        }
    }
    /**
   * Removes an account from registry
   * @param {string} accountId MetaTrader account id to remove
   */ remove(accountId) {
        delete this._rpcConnections[accountId];
        delete this._rpcConnectionInstances[accountId];
        delete this._streamingConnections[accountId];
        delete this._streamingConnectionInstances[accountId];
    }
    /**
   * Returns application type
   * @return {String} application type
   */ get application() {
        return this._application;
    }
    async _closeLastConnection(account) {
        const accountRegions = account.accountRegions;
        await Promise.all(Object.values(accountRegions).map((replicaId)=>this._metaApiWebsocketClient.unsubscribe(replicaId)));
    }
    /**
   * Returns the dictionary of streaming connections
   * @returns {{[id: string]: StreamingMetaApiConnection}}
   */ get streamingConnections() {
        return this._streamingConnections;
    }
    /**
   * Returns the dictionary of rpc connections
   * @returns {{[id: string]: RpcMetaApiConnection}}
   */ get rpcConnections() {
        return this._rpcConnections;
    }
    /**
   * Closes all connection instances for an account
   * @param {string} accountId 
   */ closeAllInstances(accountId) {
        if (this._rpcConnectionInstances[accountId]) {
            this._rpcConnectionInstances[accountId].forEach((instance)=>instance.close());
        }
        if (this._streamingConnectionInstances[accountId]) {
            this._streamingConnectionInstances[accountId].forEach((instance)=>instance.close());
        }
    }
    /**
   * Constructs a MetaTrader connection registry instance
   * @param {MetaApiOpts} options metaapi options
   * @param {MetaApiWebsocketClient} metaApiWebsocketClient MetaApi websocket client
   * @param {ClientApiClient} clientApiClient client API client
   * @param {String} application application id
   * @param {String} refreshSubscriptionsOpts subscriptions refresh options
   */ constructor(options, metaApiWebsocketClient, terminalHashManager, application = "MetaApi", refreshSubscriptionsOpts){
        refreshSubscriptionsOpts = refreshSubscriptionsOpts || {};
        this._metaApiWebsocketClient = metaApiWebsocketClient;
        this._terminalHashManager = terminalHashManager;
        this._application = application;
        this._refreshSubscriptionsOpts = refreshSubscriptionsOpts;
        this._rpcConnections = {};
        this._rpcConnectionInstances = {};
        this._streamingConnections = {};
        this._streamingConnectionInstances = {};
        this._connectionLocks = {};
        this._options = options;
    }
};

//# sourceMappingURL=data:application/json;base64,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