"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "default", {
    enumerable: true,
    get: function() {
        return MemoryHistoryStorage;
    }
});
const _historyStorage = /*#__PURE__*/ _interop_require_default(require("./historyStorage"));
const _index = /*#__PURE__*/ _interop_require_default(require("./historyDatabase/index"));
const _binarysearchtree = require("binary-search-tree");
const _logger = /*#__PURE__*/ _interop_require_default(require("../logger"));
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
let MemoryHistoryStorage = class MemoryHistoryStorage extends _historyStorage.default {
    /**
   * Initializes the storage and loads required data from a persistent storage
   * @param {string} accountId account id
   * @param {string} [application] application. Default is MetaApi
   */ async initialize(accountId, application = "MetaApi") {
        await super.initialize(accountId, application);
        let { deals, historyOrders } = await this._historyDatabase.loadHistory(accountId, application);
        for (let deal of deals){
            await this._addDeal(deal, true);
        }
        for (let historyOrder of historyOrders){
            await this._addHistoryOrder(historyOrder, true);
        }
    }
    /**
   * Resets the storage. Intended for use in tests
   * @returns {Promise} promise when the history is removed
   */ async clear() {
        this._reset();
        await this._historyDatabase.clear(this._accountId, this._application);
    }
    /**
   * Returns the time of the last history order record stored in the history storage
   * @param {Number} [instanceNumber] index of an account instance connected
   * @returns {Date} the time of the last history order record stored in the history storage
   */ lastHistoryOrderTime(instanceNumber) {
        return this._maxHistoryOrderTime;
    }
    /**
   * Returns the time of the last history deal record stored in the history storage
   * @param {Number} [instanceNumber] index of an account instance connected
   * @returns {Date} the time of the last history deal record stored in the history storage
   */ lastDealTime(instanceNumber) {
        return this._maxDealTime;
    }
    /**
   * Invoked when a new MetaTrader history order is added
   * @param {String} instanceIndex index of an account instance connected
   * @param {MetatraderOrder} historyOrder new MetaTrader history order
   */ async onHistoryOrderAdded(instanceIndex, historyOrder) {
        await this._addHistoryOrder(historyOrder);
    }
    /**
   * Invoked when a new MetaTrader history deal is added
   * @param {String} instanceIndex index of an account instance connected
   * @param {MetatraderDeal} deal new MetaTrader history deal
   */ async onDealAdded(instanceIndex, deal) {
        await this._addDeal(deal);
    }
    /**
   * Returns all deals
   * @returns {Array<MetatraderDeal>} all deals
   */ get deals() {
        return this.getDealsByTimeRange(new Date(0), new Date(8640000000000000));
    }
    /**
   * Returns deals by ticket id
   * @param {string} id ticket id
   * @returns {Array<MetatraderDeal>} deals found
   */ getDealsByTicket(id) {
        let deals = Object.values(this._dealsByTicket[id] || {});
        deals.sort(this._dealsComparator);
        return deals;
    }
    /**
   * Returns deals by position id
   * @param {string} positionId position id
   * @returns {Array<MetatraderDeal>} deals found
   */ getDealsByPosition(positionId) {
        let deals = Object.values(this._dealsByPosition[positionId] || {});
        deals.sort(this._dealsComparator);
        return deals;
    }
    /**
   * Returns deals by time range
   * @param startTime start time, inclusive
   * @param endTime end time, inclusive
   * @returns {Array<MetatraderDeal>} deals found
   */ getDealsByTimeRange(startTime, endTime) {
        let deals = this._dealsByTime.betweenBounds({
            $gte: {
                time: startTime,
                id: 0,
                entryType: ""
            },
            $lte: {
                time: endTime,
                id: Number.MAX_VALUE,
                entryType: ""
            }
        });
        return deals;
    }
    /**
   * Returns all history orders
   * @returns {Array<MetatraderOrder>} all history orders
   */ get historyOrders() {
        return this.getHistoryOrdersByTimeRange(new Date(0), new Date(8640000000000000));
    }
    /**
   * Returns history orders by ticket id
   * @param {string} id ticket id
   * @returns {Array<MetatraderOrder>} history orders found
   */ getHistoryOrdersByTicket(id) {
        let historyOrders = Object.values(this._historyOrdersByTicket[id] || {});
        historyOrders.sort(this._historyOrdersComparator);
        return historyOrders;
    }
    /**
   * Returns history orders by position id
   * @param {string} positionId position id
   * @returns {Array<MetatraderOrder>} history orders found
   */ getHistoryOrdersByPosition(positionId) {
        let historyOrders = Object.values(this._historyOrdersByPosition[positionId] || {});
        historyOrders.sort(this._historyOrdersComparator);
        return historyOrders;
    }
    /**
   * Returns history orders by time range
   * @param startTime start time, inclusive
   * @param endTime end time, inclusive
   * @returns {Array<MetatraderOrder>} hisotry orders found
   */ getHistoryOrdersByTimeRange(startTime, endTime) {
        let historyOrders = this._historyOrdersByTime.betweenBounds({
            $gte: {
                doneTime: startTime,
                id: 0,
                type: "",
                state: ""
            },
            $lte: {
                doneTime: endTime,
                id: Number.MAX_VALUE,
                type: "",
                state: ""
            }
        });
        return historyOrders;
    }
    /**
   * Invoked when a synchronization of history deals on a MetaTrader account have finished to indicate progress of an
   * initial terminal state synchronization
   * @param {String} instanceIndex index of an account instance connected
   * @param {String} synchronizationId synchronization request id
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */ async onDealsSynchronized(instanceIndex, synchronizationId) {
        await this._flushDatabase();
        await super.onDealsSynchronized(instanceIndex, synchronizationId);
    }
    _reset() {
        this._orderSynchronizationFinished = {};
        this._dealSynchronizationFinished = {};
        this._dealsByTicket = {};
        this._dealsByPosition = {};
        this._historyOrdersByTicket = {};
        this._historyOrdersByPosition = {};
        // eslint-disable-next-line complexity
        this._historyOrdersComparator = (o1, o2)=>{
            let timeDiff = (o1.doneTime || new Date(0)).getTime() - (o2.doneTime || new Date(0)).getTime();
            if (timeDiff === 0) {
                let idDiff = o1.id - o2.id;
                if (idDiff === 0) {
                    if (o1.type > o2.type) {
                        return 1;
                    } else if (o1.type < o2.type) {
                        return -1;
                    } else {
                        if (o1.state > o2.state) {
                            return 1;
                        } else if (o1.state < o2.state) {
                            return -1;
                        } else {
                            return 0;
                        }
                    }
                } else {
                    return idDiff;
                }
            } else {
                return timeDiff;
            }
        };
        this._historyOrdersByTime = new _binarysearchtree.AVLTree({
            compareKeys: this._historyOrdersComparator
        });
        this._dealsComparator = (d1, d2)=>{
            let timeDiff = (d1.time || new Date(0)).getTime() - (d2.time || new Date(0)).getTime();
            if (timeDiff === 0) {
                let idDiff = d1.id - d2.id;
                if (idDiff === 0) {
                    if (d1.entryType > d2.entryType) {
                        return 1;
                    } else if (d1.entryType < d2.entryType) {
                        return -1;
                    } else {
                        return 0;
                    }
                } else {
                    return idDiff;
                }
            } else {
                return timeDiff;
            }
        };
        this._dealsByTime = new _binarysearchtree.AVLTree({
            compareKeys: this._dealsComparator
        });
        this._maxHistoryOrderTime = new Date(0);
        this._maxDealTime = new Date(0);
        this._newHistoryOrders = [];
        this._newDeals = [];
        clearTimeout(this._flushTimeout);
        delete this._flushTimeout;
    }
    // eslint-disable-next-line complexity
    async _addDeal(deal, existing) {
        let key = this._getDealKey(deal);
        this._dealsByTicket[deal.id] = this._dealsByTicket[deal.id] || {};
        let newDeal = !existing && !this._dealsByTicket[deal.id][key];
        this._dealsByTicket[deal.id][key] = deal;
        if (deal.positionId) {
            this._dealsByPosition[deal.positionId] = this._dealsByPosition[deal.positionId] || {};
            this._dealsByPosition[deal.positionId][key] = deal;
        }
        this._dealsByTime.delete(deal);
        this._dealsByTime.insert(deal, deal);
        if (deal.time && (!this._maxDealTime || this._maxDealTime.getTime() < deal.time.getTime())) {
            this._maxDealTime = deal.time;
        }
        if (newDeal) {
            this._newDeals.push(deal);
            clearTimeout(this._flushTimeout);
            this._flushTimeout = setTimeout(this._flushDatabase.bind(this), 5000);
        }
    }
    _getDealKey(deal) {
        return (deal.time || new Date(0)).toISOString() + ":" + deal.id + ":" + deal.entryType;
    }
    // eslint-disable-next-line complexity
    async _addHistoryOrder(historyOrder, existing) {
        let key = this._getHistoryOrderKey(historyOrder);
        this._historyOrdersByTicket[historyOrder.id] = this._historyOrdersByTicket[historyOrder.id] || {};
        let newHistoryOrder = !existing && !this._historyOrdersByTicket[historyOrder.id][key];
        this._historyOrdersByTicket[historyOrder.id][key] = historyOrder;
        if (historyOrder.positionId) {
            this._historyOrdersByPosition[historyOrder.positionId] = this._historyOrdersByPosition[historyOrder.positionId] || {};
            this._historyOrdersByPosition[historyOrder.positionId][key] = historyOrder;
        }
        this._historyOrdersByTime.delete(historyOrder);
        this._historyOrdersByTime.insert(historyOrder, historyOrder);
        if (historyOrder.doneTime && (!this._maxHistoryOrderTime || this._maxHistoryOrderTime.getTime() < historyOrder.doneTime.getTime())) {
            this._maxHistoryOrderTime = historyOrder.doneTime;
        }
        if (newHistoryOrder) {
            this._newHistoryOrders.push(historyOrder);
            clearTimeout(this._flushTimeout);
            this._flushTimeout = setTimeout(this._flushDatabase.bind(this), 5000);
        }
    }
    _getHistoryOrderKey(historyOrder) {
        return (historyOrder.doneTime || new Date(0)).toISOString() + ":" + historyOrder.id + ":" + historyOrder.type + ":" + historyOrder.status;
    }
    async _flushDatabase() {
        if (this._flushPromise) {
            await this._flushPromise;
        }
        if (this._flushRunning) {
            return;
        }
        this._flushRunning = true;
        let resolve;
        this._flushPromise = new Promise((res)=>resolve = res);
        try {
            await this._historyDatabase.flush(this._accountId, this._application, this._newHistoryOrders, this._newDeals);
            this._newHistoryOrders = [];
            this._newDeals = [];
            this._logger.debug(`${this._accountId}: flushed history db`);
        } catch (err) {
            this._logger.warn(`${this._accountId}: error flushing history db`, err);
            this._flushTimeout = setTimeout(this._flushDatabase.bind(this), 15000);
        } finally{
            resolve();
            this._flushRunning = false;
        }
    }
    /**
   * Constructs the in-memory history store instance
   */ constructor(){
        super();
        this._historyDatabase = _index.default.getInstance();
        this._reset();
        this._logger = _logger.default.getLogger("MemoryHistoryStorage");
    }
};

//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIjxhbm9uPiJdLCJzb3VyY2VzQ29udGVudCI6WyIndXNlIHN0cmljdCc7XG5cbmltcG9ydCBIaXN0b3J5U3RvcmFnZSBmcm9tICcuL2hpc3RvcnlTdG9yYWdlJztcbmltcG9ydCBIaXN0b3J5RGF0YWJhc2UgZnJvbSAnLi9oaXN0b3J5RGF0YWJhc2UvaW5kZXgnO1xuaW1wb3J0IHsgQVZMVHJlZSB9IGZyb20gJ2JpbmFyeS1zZWFyY2gtdHJlZSc7XG5pbXBvcnQgTG9nZ2VyTWFuYWdlciBmcm9tICcuLi9sb2dnZXInO1xuXG4vKipcbiAqIEhpc3Rvcnkgc3RvcmFnZSB3aGljaCBzdG9yZXMgTWV0YVRyYWRlciBoaXN0b3J5IGluIFJBTVxuICovXG5leHBvcnQgZGVmYXVsdCBjbGFzcyBNZW1vcnlIaXN0b3J5U3RvcmFnZSBleHRlbmRzIEhpc3RvcnlTdG9yYWdlIHtcblxuICAvKipcbiAgICogQ29uc3RydWN0cyB0aGUgaW4tbWVtb3J5IGhpc3Rvcnkgc3RvcmUgaW5zdGFuY2VcbiAgICovXG4gIGNvbnN0cnVjdG9yKCkge1xuICAgIHN1cGVyKCk7XG4gICAgdGhpcy5faGlzdG9yeURhdGFiYXNlID0gSGlzdG9yeURhdGFiYXNlLmdldEluc3RhbmNlKCk7XG4gICAgdGhpcy5fcmVzZXQoKTtcbiAgICB0aGlzLl9sb2dnZXIgPSBMb2dnZXJNYW5hZ2VyLmdldExvZ2dlcignTWVtb3J5SGlzdG9yeVN0b3JhZ2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBJbml0aWFsaXplcyB0aGUgc3RvcmFnZSBhbmQgbG9hZHMgcmVxdWlyZWQgZGF0YSBmcm9tIGEgcGVyc2lzdGVudCBzdG9yYWdlXG4gICAqIEBwYXJhbSB7c3RyaW5nfSBhY2NvdW50SWQgYWNjb3VudCBpZFxuICAgKiBAcGFyYW0ge3N0cmluZ30gW2FwcGxpY2F0aW9uXSBhcHBsaWNhdGlvbi4gRGVmYXVsdCBpcyBNZXRhQXBpXG4gICAqL1xuICBhc3luYyBpbml0aWFsaXplKGFjY291bnRJZCwgYXBwbGljYXRpb24gPSAnTWV0YUFwaScpIHtcbiAgICBhd2FpdCBzdXBlci5pbml0aWFsaXplKGFjY291bnRJZCwgYXBwbGljYXRpb24pO1xuICAgIGxldCB7ZGVhbHMsIGhpc3RvcnlPcmRlcnN9ID0gYXdhaXQgdGhpcy5faGlzdG9yeURhdGFiYXNlLmxvYWRIaXN0b3J5KGFjY291bnRJZCwgYXBwbGljYXRpb24pO1xuICAgIGZvciAobGV0IGRlYWwgb2YgZGVhbHMpIHtcbiAgICAgIGF3YWl0IHRoaXMuX2FkZERlYWwoZGVhbCwgdHJ1ZSk7XG4gICAgfVxuICAgIGZvciAobGV0IGhpc3RvcnlPcmRlciBvZiBoaXN0b3J5T3JkZXJzKSB7XG4gICAgICBhd2FpdCB0aGlzLl9hZGRIaXN0b3J5T3JkZXIoaGlzdG9yeU9yZGVyLCB0cnVlKTtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogUmVzZXRzIHRoZSBzdG9yYWdlLiBJbnRlbmRlZCBmb3IgdXNlIGluIHRlc3RzXG4gICAqIEByZXR1cm5zIHtQcm9taXNlfSBwcm9taXNlIHdoZW4gdGhlIGhpc3RvcnkgaXMgcmVtb3ZlZFxuICAgKi9cbiAgYXN5bmMgY2xlYXIoKSB7XG4gICAgdGhpcy5fcmVzZXQoKTtcbiAgICBhd2FpdCB0aGlzLl9oaXN0b3J5RGF0YWJhc2UuY2xlYXIodGhpcy5fYWNjb3VudElkLCB0aGlzLl9hcHBsaWNhdGlvbik7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyB0aGUgdGltZSBvZiB0aGUgbGFzdCBoaXN0b3J5IG9yZGVyIHJlY29yZCBzdG9yZWQgaW4gdGhlIGhpc3Rvcnkgc3RvcmFnZVxuICAgKiBAcGFyYW0ge051bWJlcn0gW2luc3RhbmNlTnVtYmVyXSBpbmRleCBvZiBhbiBhY2NvdW50IGluc3RhbmNlIGNvbm5lY3RlZFxuICAgKiBAcmV0dXJucyB7RGF0ZX0gdGhlIHRpbWUgb2YgdGhlIGxhc3QgaGlzdG9yeSBvcmRlciByZWNvcmQgc3RvcmVkIGluIHRoZSBoaXN0b3J5IHN0b3JhZ2VcbiAgICovXG4gIGxhc3RIaXN0b3J5T3JkZXJUaW1lKGluc3RhbmNlTnVtYmVyKSB7XG4gICAgcmV0dXJuIHRoaXMuX21heEhpc3RvcnlPcmRlclRpbWU7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyB0aGUgdGltZSBvZiB0aGUgbGFzdCBoaXN0b3J5IGRlYWwgcmVjb3JkIHN0b3JlZCBpbiB0aGUgaGlzdG9yeSBzdG9yYWdlXG4gICAqIEBwYXJhbSB7TnVtYmVyfSBbaW5zdGFuY2VOdW1iZXJdIGluZGV4IG9mIGFuIGFjY291bnQgaW5zdGFuY2UgY29ubmVjdGVkXG4gICAqIEByZXR1cm5zIHtEYXRlfSB0aGUgdGltZSBvZiB0aGUgbGFzdCBoaXN0b3J5IGRlYWwgcmVjb3JkIHN0b3JlZCBpbiB0aGUgaGlzdG9yeSBzdG9yYWdlXG4gICAqL1xuICBsYXN0RGVhbFRpbWUoaW5zdGFuY2VOdW1iZXIpIHtcbiAgICByZXR1cm4gdGhpcy5fbWF4RGVhbFRpbWU7XG4gIH1cblxuICAvKipcbiAgICogSW52b2tlZCB3aGVuIGEgbmV3IE1ldGFUcmFkZXIgaGlzdG9yeSBvcmRlciBpcyBhZGRlZFxuICAgKiBAcGFyYW0ge1N0cmluZ30gaW5zdGFuY2VJbmRleCBpbmRleCBvZiBhbiBhY2NvdW50IGluc3RhbmNlIGNvbm5lY3RlZFxuICAgKiBAcGFyYW0ge01ldGF0cmFkZXJPcmRlcn0gaGlzdG9yeU9yZGVyIG5ldyBNZXRhVHJhZGVyIGhpc3Rvcnkgb3JkZXJcbiAgICovXG4gIGFzeW5jIG9uSGlzdG9yeU9yZGVyQWRkZWQoaW5zdGFuY2VJbmRleCwgaGlzdG9yeU9yZGVyKSB7XG4gICAgYXdhaXQgdGhpcy5fYWRkSGlzdG9yeU9yZGVyKGhpc3RvcnlPcmRlcik7XG4gIH1cblxuICAvKipcbiAgICogSW52b2tlZCB3aGVuIGEgbmV3IE1ldGFUcmFkZXIgaGlzdG9yeSBkZWFsIGlzIGFkZGVkXG4gICAqIEBwYXJhbSB7U3RyaW5nfSBpbnN0YW5jZUluZGV4IGluZGV4IG9mIGFuIGFjY291bnQgaW5zdGFuY2UgY29ubmVjdGVkXG4gICAqIEBwYXJhbSB7TWV0YXRyYWRlckRlYWx9IGRlYWwgbmV3IE1ldGFUcmFkZXIgaGlzdG9yeSBkZWFsXG4gICAqL1xuICBhc3luYyBvbkRlYWxBZGRlZChpbnN0YW5jZUluZGV4LCBkZWFsKSB7XG4gICAgYXdhaXQgdGhpcy5fYWRkRGVhbChkZWFsKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIGFsbCBkZWFsc1xuICAgKiBAcmV0dXJucyB7QXJyYXk8TWV0YXRyYWRlckRlYWw+fSBhbGwgZGVhbHNcbiAgICovXG4gIGdldCBkZWFscygpIHtcbiAgICByZXR1cm4gdGhpcy5nZXREZWFsc0J5VGltZVJhbmdlKG5ldyBEYXRlKDApLCBuZXcgRGF0ZSg4NjQwMDAwMDAwMDAwMDAwKSk7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyBkZWFscyBieSB0aWNrZXQgaWRcbiAgICogQHBhcmFtIHtzdHJpbmd9IGlkIHRpY2tldCBpZFxuICAgKiBAcmV0dXJucyB7QXJyYXk8TWV0YXRyYWRlckRlYWw+fSBkZWFscyBmb3VuZFxuICAgKi9cbiAgZ2V0RGVhbHNCeVRpY2tldChpZCkge1xuICAgIGxldCBkZWFscyA9IE9iamVjdC52YWx1ZXModGhpcy5fZGVhbHNCeVRpY2tldFtpZF0gfHwge30pO1xuICAgIGRlYWxzLnNvcnQodGhpcy5fZGVhbHNDb21wYXJhdG9yKTtcbiAgICByZXR1cm4gZGVhbHM7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyBkZWFscyBieSBwb3NpdGlvbiBpZFxuICAgKiBAcGFyYW0ge3N0cmluZ30gcG9zaXRpb25JZCBwb3NpdGlvbiBpZFxuICAgKiBAcmV0dXJucyB7QXJyYXk8TWV0YXRyYWRlckRlYWw+fSBkZWFscyBmb3VuZFxuICAgKi9cbiAgZ2V0RGVhbHNCeVBvc2l0aW9uKHBvc2l0aW9uSWQpIHtcbiAgICBsZXQgZGVhbHMgPSBPYmplY3QudmFsdWVzKHRoaXMuX2RlYWxzQnlQb3NpdGlvbltwb3NpdGlvbklkXSB8fCB7fSk7XG4gICAgZGVhbHMuc29ydCh0aGlzLl9kZWFsc0NvbXBhcmF0b3IpO1xuICAgIHJldHVybiBkZWFscztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIGRlYWxzIGJ5IHRpbWUgcmFuZ2VcbiAgICogQHBhcmFtIHN0YXJ0VGltZSBzdGFydCB0aW1lLCBpbmNsdXNpdmVcbiAgICogQHBhcmFtIGVuZFRpbWUgZW5kIHRpbWUsIGluY2x1c2l2ZVxuICAgKiBAcmV0dXJucyB7QXJyYXk8TWV0YXRyYWRlckRlYWw+fSBkZWFscyBmb3VuZFxuICAgKi9cbiAgZ2V0RGVhbHNCeVRpbWVSYW5nZShzdGFydFRpbWUsIGVuZFRpbWUpIHtcbiAgICBsZXQgZGVhbHMgPSB0aGlzLl9kZWFsc0J5VGltZS5iZXR3ZWVuQm91bmRzKHtcbiAgICAgICRndGU6IHt0aW1lOiBzdGFydFRpbWUsIGlkOiAwLCBlbnRyeVR5cGU6ICcnfSxcbiAgICAgICRsdGU6IHt0aW1lOiBlbmRUaW1lLCBpZDogTnVtYmVyLk1BWF9WQUxVRSwgZW50cnlUeXBlOiAnJ31cbiAgICB9KTtcbiAgICByZXR1cm4gZGVhbHM7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyBhbGwgaGlzdG9yeSBvcmRlcnNcbiAgICogQHJldHVybnMge0FycmF5PE1ldGF0cmFkZXJPcmRlcj59IGFsbCBoaXN0b3J5IG9yZGVyc1xuICAgKi9cbiAgZ2V0IGhpc3RvcnlPcmRlcnMoKSB7XG4gICAgcmV0dXJuIHRoaXMuZ2V0SGlzdG9yeU9yZGVyc0J5VGltZVJhbmdlKG5ldyBEYXRlKDApLCBuZXcgRGF0ZSg4NjQwMDAwMDAwMDAwMDAwKSk7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyBoaXN0b3J5IG9yZGVycyBieSB0aWNrZXQgaWRcbiAgICogQHBhcmFtIHtzdHJpbmd9IGlkIHRpY2tldCBpZFxuICAgKiBAcmV0dXJucyB7QXJyYXk8TWV0YXRyYWRlck9yZGVyPn0gaGlzdG9yeSBvcmRlcnMgZm91bmRcbiAgICovXG4gIGdldEhpc3RvcnlPcmRlcnNCeVRpY2tldChpZCkge1xuICAgIGxldCBoaXN0b3J5T3JkZXJzID0gT2JqZWN0LnZhbHVlcyh0aGlzLl9oaXN0b3J5T3JkZXJzQnlUaWNrZXRbaWRdIHx8IHt9KTtcbiAgICBoaXN0b3J5T3JkZXJzLnNvcnQodGhpcy5faGlzdG9yeU9yZGVyc0NvbXBhcmF0b3IpO1xuICAgIHJldHVybiBoaXN0b3J5T3JkZXJzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgaGlzdG9yeSBvcmRlcnMgYnkgcG9zaXRpb24gaWRcbiAgICogQHBhcmFtIHtzdHJpbmd9IHBvc2l0aW9uSWQgcG9zaXRpb24gaWRcbiAgICogQHJldHVybnMge0FycmF5PE1ldGF0cmFkZXJPcmRlcj59IGhpc3Rvcnkgb3JkZXJzIGZvdW5kXG4gICAqL1xuICBnZXRIaXN0b3J5T3JkZXJzQnlQb3NpdGlvbihwb3NpdGlvbklkKSB7XG4gICAgbGV0IGhpc3RvcnlPcmRlcnMgPSBPYmplY3QudmFsdWVzKHRoaXMuX2hpc3RvcnlPcmRlcnNCeVBvc2l0aW9uW3Bvc2l0aW9uSWRdIHx8IHt9KTtcbiAgICBoaXN0b3J5T3JkZXJzLnNvcnQodGhpcy5faGlzdG9yeU9yZGVyc0NvbXBhcmF0b3IpO1xuICAgIHJldHVybiBoaXN0b3J5T3JkZXJzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgaGlzdG9yeSBvcmRlcnMgYnkgdGltZSByYW5nZVxuICAgKiBAcGFyYW0gc3RhcnRUaW1lIHN0YXJ0IHRpbWUsIGluY2x1c2l2ZVxuICAgKiBAcGFyYW0gZW5kVGltZSBlbmQgdGltZSwgaW5jbHVzaXZlXG4gICAqIEByZXR1cm5zIHtBcnJheTxNZXRhdHJhZGVyT3JkZXI+fSBoaXNvdHJ5IG9yZGVycyBmb3VuZFxuICAgKi9cbiAgZ2V0SGlzdG9yeU9yZGVyc0J5VGltZVJhbmdlKHN0YXJ0VGltZSwgZW5kVGltZSkge1xuICAgIGxldCBoaXN0b3J5T3JkZXJzID0gdGhpcy5faGlzdG9yeU9yZGVyc0J5VGltZS5iZXR3ZWVuQm91bmRzKHtcbiAgICAgICRndGU6IHtkb25lVGltZTogc3RhcnRUaW1lLCBpZDogMCwgdHlwZTogJycsIHN0YXRlOiAnJ30sXG4gICAgICAkbHRlOiB7ZG9uZVRpbWU6IGVuZFRpbWUsIGlkOiBOdW1iZXIuTUFYX1ZBTFVFLCB0eXBlOiAnJywgc3RhdGU6ICcnfVxuICAgIH0pO1xuICAgIHJldHVybiBoaXN0b3J5T3JkZXJzO1xuICB9XG5cbiAgLyoqXG4gICAqIEludm9rZWQgd2hlbiBhIHN5bmNocm9uaXphdGlvbiBvZiBoaXN0b3J5IGRlYWxzIG9uIGEgTWV0YVRyYWRlciBhY2NvdW50IGhhdmUgZmluaXNoZWQgdG8gaW5kaWNhdGUgcHJvZ3Jlc3Mgb2YgYW5cbiAgICogaW5pdGlhbCB0ZXJtaW5hbCBzdGF0ZSBzeW5jaHJvbml6YXRpb25cbiAgICogQHBhcmFtIHtTdHJpbmd9IGluc3RhbmNlSW5kZXggaW5kZXggb2YgYW4gYWNjb3VudCBpbnN0YW5jZSBjb25uZWN0ZWRcbiAgICogQHBhcmFtIHtTdHJpbmd9IHN5bmNocm9uaXphdGlvbklkIHN5bmNocm9uaXphdGlvbiByZXF1ZXN0IGlkXG4gICAqIEByZXR1cm4ge1Byb21pc2V9IHByb21pc2Ugd2hpY2ggcmVzb2x2ZXMgd2hlbiB0aGUgYXN5bmNocm9ub3VzIGV2ZW50IGlzIHByb2Nlc3NlZFxuICAgKi9cbiAgYXN5bmMgb25EZWFsc1N5bmNocm9uaXplZChpbnN0YW5jZUluZGV4LCBzeW5jaHJvbml6YXRpb25JZCkge1xuICAgIGF3YWl0IHRoaXMuX2ZsdXNoRGF0YWJhc2UoKTtcbiAgICBhd2FpdCBzdXBlci5vbkRlYWxzU3luY2hyb25pemVkKGluc3RhbmNlSW5kZXgsIHN5bmNocm9uaXphdGlvbklkKTtcbiAgfVxuXG4gIF9yZXNldCgpIHtcbiAgICB0aGlzLl9vcmRlclN5bmNocm9uaXphdGlvbkZpbmlzaGVkID0ge307XG4gICAgdGhpcy5fZGVhbFN5bmNocm9uaXphdGlvbkZpbmlzaGVkID0ge307XG4gICAgdGhpcy5fZGVhbHNCeVRpY2tldCA9IHt9O1xuICAgIHRoaXMuX2RlYWxzQnlQb3NpdGlvbiA9IHt9O1xuICAgIHRoaXMuX2hpc3RvcnlPcmRlcnNCeVRpY2tldCA9IHt9O1xuICAgIHRoaXMuX2hpc3RvcnlPcmRlcnNCeVBvc2l0aW9uID0ge307XG4gICAgLy8gZXNsaW50LWRpc2FibGUtbmV4dC1saW5lIGNvbXBsZXhpdHlcbiAgICB0aGlzLl9oaXN0b3J5T3JkZXJzQ29tcGFyYXRvciA9IChvMSwgbzIpID0+IHtcbiAgICAgIGxldCB0aW1lRGlmZiA9IChvMS5kb25lVGltZSB8fCBuZXcgRGF0ZSgwKSkuZ2V0VGltZSgpIC0gKG8yLmRvbmVUaW1lIHx8IG5ldyBEYXRlKDApKS5nZXRUaW1lKCk7XG4gICAgICBpZiAodGltZURpZmYgPT09IDApIHtcbiAgICAgICAgbGV0IGlkRGlmZiA9IG8xLmlkIC0gbzIuaWQ7XG4gICAgICAgIGlmIChpZERpZmYgPT09IDApIHtcbiAgICAgICAgICBpZiAobzEudHlwZSA+IG8yLnR5cGUpIHtcbiAgICAgICAgICAgIHJldHVybiAxO1xuICAgICAgICAgIH0gZWxzZSBpZiAobzEudHlwZSA8IG8yLnR5cGUpIHtcbiAgICAgICAgICAgIHJldHVybiAtMTtcbiAgICAgICAgICB9IGVsc2Uge1xuICAgICAgICAgICAgaWYgKG8xLnN0YXRlID4gbzIuc3RhdGUpIHtcbiAgICAgICAgICAgICAgcmV0dXJuIDE7XG4gICAgICAgICAgICB9IGVsc2UgaWYgKG8xLnN0YXRlIDwgbzIuc3RhdGUpIHtcbiAgICAgICAgICAgICAgcmV0dXJuIC0xO1xuICAgICAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICAgICAgcmV0dXJuIDA7XG4gICAgICAgICAgICB9XG4gICAgICAgICAgfVxuICAgICAgICB9IGVsc2Uge1xuICAgICAgICAgIHJldHVybiBpZERpZmY7XG4gICAgICAgIH1cbiAgICAgIH0gZWxzZSB7XG4gICAgICAgIHJldHVybiB0aW1lRGlmZjtcbiAgICAgIH1cbiAgICB9O1xuICAgIHRoaXMuX2hpc3RvcnlPcmRlcnNCeVRpbWUgPSBuZXcgQVZMVHJlZSh7Y29tcGFyZUtleXM6IHRoaXMuX2hpc3RvcnlPcmRlcnNDb21wYXJhdG9yfSk7XG4gICAgdGhpcy5fZGVhbHNDb21wYXJhdG9yID0gKGQxLCBkMikgPT4ge1xuICAgICAgbGV0IHRpbWVEaWZmID0gKGQxLnRpbWUgfHwgbmV3IERhdGUoMCkpLmdldFRpbWUoKSAtIChkMi50aW1lIHx8IG5ldyBEYXRlKDApKS5nZXRUaW1lKCk7XG4gICAgICBpZiAodGltZURpZmYgPT09IDApIHtcbiAgICAgICAgbGV0IGlkRGlmZiA9IGQxLmlkIC0gZDIuaWQ7XG4gICAgICAgIGlmIChpZERpZmYgPT09IDApIHtcbiAgICAgICAgICBpZiAoZDEuZW50cnlUeXBlID4gZDIuZW50cnlUeXBlKSB7XG4gICAgICAgICAgICByZXR1cm4gMTtcbiAgICAgICAgICB9IGVsc2UgaWYgKGQxLmVudHJ5VHlwZSA8IGQyLmVudHJ5VHlwZSkge1xuICAgICAgICAgICAgcmV0dXJuIC0xO1xuICAgICAgICAgIH0gZWxzZSB7XG4gICAgICAgICAgICByZXR1cm4gMDtcbiAgICAgICAgICB9XG4gICAgICAgIH0gZWxzZSB7XG4gICAgICAgICAgcmV0dXJuIGlkRGlmZjtcbiAgICAgICAgfVxuICAgICAgfSBlbHNlIHtcbiAgICAgICAgcmV0dXJuIHRpbWVEaWZmO1xuICAgICAgfVxuICAgIH07XG4gICAgdGhpcy5fZGVhbHNCeVRpbWUgPSBuZXcgQVZMVHJlZSh7Y29tcGFyZUtleXM6IHRoaXMuX2RlYWxzQ29tcGFyYXRvcn0pO1xuICAgIHRoaXMuX21heEhpc3RvcnlPcmRlclRpbWUgPSBuZXcgRGF0ZSgwKTtcbiAgICB0aGlzLl9tYXhEZWFsVGltZSA9IG5ldyBEYXRlKDApO1xuICAgIHRoaXMuX25ld0hpc3RvcnlPcmRlcnMgPSBbXTtcbiAgICB0aGlzLl9uZXdEZWFscyA9IFtdO1xuICAgIGNsZWFyVGltZW91dCh0aGlzLl9mbHVzaFRpbWVvdXQpO1xuICAgIGRlbGV0ZSB0aGlzLl9mbHVzaFRpbWVvdXQ7XG4gIH1cblxuICAvLyBlc2xpbnQtZGlzYWJsZS1uZXh0LWxpbmUgY29tcGxleGl0eVxuICBhc3luYyBfYWRkRGVhbChkZWFsLCBleGlzdGluZykge1xuICAgIGxldCBrZXkgPSB0aGlzLl9nZXREZWFsS2V5KGRlYWwpO1xuICAgIHRoaXMuX2RlYWxzQnlUaWNrZXRbZGVhbC5pZF0gPSB0aGlzLl9kZWFsc0J5VGlja2V0W2RlYWwuaWRdIHx8IHt9O1xuICAgIGxldCBuZXdEZWFsID0gIWV4aXN0aW5nICYmICF0aGlzLl9kZWFsc0J5VGlja2V0W2RlYWwuaWRdW2tleV07XG4gICAgdGhpcy5fZGVhbHNCeVRpY2tldFtkZWFsLmlkXVtrZXldID0gZGVhbDtcbiAgICBpZiAoZGVhbC5wb3NpdGlvbklkKSB7XG4gICAgICB0aGlzLl9kZWFsc0J5UG9zaXRpb25bZGVhbC5wb3NpdGlvbklkXSA9IHRoaXMuX2RlYWxzQnlQb3NpdGlvbltkZWFsLnBvc2l0aW9uSWRdIHx8IHt9O1xuICAgICAgdGhpcy5fZGVhbHNCeVBvc2l0aW9uW2RlYWwucG9zaXRpb25JZF1ba2V5XSA9IGRlYWw7XG4gICAgfVxuICAgIHRoaXMuX2RlYWxzQnlUaW1lLmRlbGV0ZShkZWFsKTtcbiAgICB0aGlzLl9kZWFsc0J5VGltZS5pbnNlcnQoZGVhbCwgZGVhbCk7XG4gICAgaWYgKGRlYWwudGltZSAmJiAoIXRoaXMuX21heERlYWxUaW1lIHx8IHRoaXMuX21heERlYWxUaW1lLmdldFRpbWUoKSA8IGRlYWwudGltZS5nZXRUaW1lKCkpKSB7XG4gICAgICB0aGlzLl9tYXhEZWFsVGltZSA9IGRlYWwudGltZTtcbiAgICB9XG4gICAgaWYgKG5ld0RlYWwpIHtcbiAgICAgIHRoaXMuX25ld0RlYWxzLnB1c2goZGVhbCk7XG4gICAgICBjbGVhclRpbWVvdXQodGhpcy5fZmx1c2hUaW1lb3V0KTtcbiAgICAgIHRoaXMuX2ZsdXNoVGltZW91dCA9IHNldFRpbWVvdXQodGhpcy5fZmx1c2hEYXRhYmFzZS5iaW5kKHRoaXMpLCA1MDAwKTtcbiAgICB9XG4gIH1cblxuICBfZ2V0RGVhbEtleShkZWFsKSB7XG4gICAgcmV0dXJuIChkZWFsLnRpbWUgfHwgbmV3IERhdGUoMCkpLnRvSVNPU3RyaW5nKCkgKyAnOicgKyBkZWFsLmlkICsgJzonICsgZGVhbC5lbnRyeVR5cGU7XG4gIH1cblxuICAvLyBlc2xpbnQtZGlzYWJsZS1uZXh0LWxpbmUgY29tcGxleGl0eVxuICBhc3luYyBfYWRkSGlzdG9yeU9yZGVyKGhpc3RvcnlPcmRlciwgZXhpc3RpbmcpIHtcbiAgICBsZXQga2V5ID0gdGhpcy5fZ2V0SGlzdG9yeU9yZGVyS2V5KGhpc3RvcnlPcmRlcik7XG4gICAgdGhpcy5faGlzdG9yeU9yZGVyc0J5VGlja2V0W2hpc3RvcnlPcmRlci5pZF0gPSB0aGlzLl9oaXN0b3J5T3JkZXJzQnlUaWNrZXRbaGlzdG9yeU9yZGVyLmlkXSB8fCB7fTtcbiAgICBsZXQgbmV3SGlzdG9yeU9yZGVyID0gIWV4aXN0aW5nICYmICF0aGlzLl9oaXN0b3J5T3JkZXJzQnlUaWNrZXRbaGlzdG9yeU9yZGVyLmlkXVtrZXldO1xuICAgIHRoaXMuX2hpc3RvcnlPcmRlcnNCeVRpY2tldFtoaXN0b3J5T3JkZXIuaWRdW2tleV0gPSBoaXN0b3J5T3JkZXI7XG4gICAgaWYgKGhpc3RvcnlPcmRlci5wb3NpdGlvbklkKSB7XG4gICAgICB0aGlzLl9oaXN0b3J5T3JkZXJzQnlQb3NpdGlvbltoaXN0b3J5T3JkZXIucG9zaXRpb25JZF0gPSB0aGlzLl9oaXN0b3J5T3JkZXJzQnlQb3NpdGlvbltoaXN0b3J5T3JkZXIucG9zaXRpb25JZF0gfHxcbiAgICAgICAge307XG4gICAgICB0aGlzLl9oaXN0b3J5T3JkZXJzQnlQb3NpdGlvbltoaXN0b3J5T3JkZXIucG9zaXRpb25JZF1ba2V5XSA9IGhpc3RvcnlPcmRlcjtcbiAgICB9XG4gICAgdGhpcy5faGlzdG9yeU9yZGVyc0J5VGltZS5kZWxldGUoaGlzdG9yeU9yZGVyKTtcbiAgICB0aGlzLl9oaXN0b3J5T3JkZXJzQnlUaW1lLmluc2VydChoaXN0b3J5T3JkZXIsIGhpc3RvcnlPcmRlcik7XG4gICAgaWYgKGhpc3RvcnlPcmRlci5kb25lVGltZSAmJiAoIXRoaXMuX21heEhpc3RvcnlPcmRlclRpbWUgfHxcbiAgICAgICAgdGhpcy5fbWF4SGlzdG9yeU9yZGVyVGltZS5nZXRUaW1lKCkgPCBoaXN0b3J5T3JkZXIuZG9uZVRpbWUuZ2V0VGltZSgpKSkge1xuICAgICAgdGhpcy5fbWF4SGlzdG9yeU9yZGVyVGltZSA9IGhpc3RvcnlPcmRlci5kb25lVGltZTtcbiAgICB9XG4gICAgaWYgKG5ld0hpc3RvcnlPcmRlcikge1xuICAgICAgdGhpcy5fbmV3SGlzdG9yeU9yZGVycy5wdXNoKGhpc3RvcnlPcmRlcik7XG4gICAgICBjbGVhclRpbWVvdXQodGhpcy5fZmx1c2hUaW1lb3V0KTtcbiAgICAgIHRoaXMuX2ZsdXNoVGltZW91dCA9IHNldFRpbWVvdXQodGhpcy5fZmx1c2hEYXRhYmFzZS5iaW5kKHRoaXMpLCA1MDAwKTtcbiAgICB9XG4gIH1cblxuICBfZ2V0SGlzdG9yeU9yZGVyS2V5KGhpc3RvcnlPcmRlcikge1xuICAgIHJldHVybiAoaGlzdG9yeU9yZGVyLmRvbmVUaW1lIHx8IG5ldyBEYXRlKDApKS50b0lTT1N0cmluZygpICsgJzonICsgaGlzdG9yeU9yZGVyLmlkICsgJzonICtcbiAgICAgIGhpc3RvcnlPcmRlci50eXBlICsgJzonICsgaGlzdG9yeU9yZGVyLnN0YXR1cztcbiAgfVxuXG4gIGFzeW5jIF9mbHVzaERhdGFiYXNlKCkge1xuICAgIGlmICh0aGlzLl9mbHVzaFByb21pc2UpIHtcbiAgICAgIGF3YWl0IHRoaXMuX2ZsdXNoUHJvbWlzZTtcbiAgICB9XG4gICAgaWYgKHRoaXMuX2ZsdXNoUnVubmluZykge1xuICAgICAgcmV0dXJuO1xuICAgIH1cbiAgICB0aGlzLl9mbHVzaFJ1bm5pbmcgPSB0cnVlO1xuICAgIGxldCByZXNvbHZlO1xuICAgIHRoaXMuX2ZsdXNoUHJvbWlzZSA9IG5ldyBQcm9taXNlKHJlcyA9PiByZXNvbHZlID0gcmVzKTtcbiAgICB0cnkge1xuICAgICAgYXdhaXQgdGhpcy5faGlzdG9yeURhdGFiYXNlLmZsdXNoKHRoaXMuX2FjY291bnRJZCwgdGhpcy5fYXBwbGljYXRpb24sIHRoaXMuX25ld0hpc3RvcnlPcmRlcnMsIHRoaXMuX25ld0RlYWxzKTtcbiAgICAgIHRoaXMuX25ld0hpc3RvcnlPcmRlcnMgPSBbXTtcbiAgICAgIHRoaXMuX25ld0RlYWxzID0gW107XG4gICAgICB0aGlzLl9sb2dnZXIuZGVidWcoYCR7dGhpcy5fYWNjb3VudElkfTogZmx1c2hlZCBoaXN0b3J5IGRiYCk7XG4gICAgfSBjYXRjaCAoZXJyKSB7XG4gICAgICB0aGlzLl9sb2dnZXIud2FybihgJHt0aGlzLl9hY2NvdW50SWR9OiBlcnJvciBmbHVzaGluZyBoaXN0b3J5IGRiYCwgZXJyKTtcbiAgICAgIHRoaXMuX2ZsdXNoVGltZW91dCA9IHNldFRpbWVvdXQodGhpcy5fZmx1c2hEYXRhYmFzZS5iaW5kKHRoaXMpLCAxNTAwMCk7XG4gICAgfSBmaW5hbGx5IHtcbiAgICAgIHJlc29sdmUoKTtcbiAgICAgIHRoaXMuX2ZsdXNoUnVubmluZyA9IGZhbHNlO1xuICAgIH1cbiAgfVxuXG59XG4iXSwibmFtZXMiOlsiTWVtb3J5SGlzdG9yeVN0b3JhZ2UiLCJIaXN0b3J5U3RvcmFnZSIsImluaXRpYWxpemUiLCJhY2NvdW50SWQiLCJhcHBsaWNhdGlvbiIsImRlYWxzIiwiaGlzdG9yeU9yZGVycyIsIl9oaXN0b3J5RGF0YWJhc2UiLCJsb2FkSGlzdG9yeSIsImRlYWwiLCJfYWRkRGVhbCIsImhpc3RvcnlPcmRlciIsIl9hZGRIaXN0b3J5T3JkZXIiLCJjbGVhciIsIl9yZXNldCIsIl9hY2NvdW50SWQiLCJfYXBwbGljYXRpb24iLCJsYXN0SGlzdG9yeU9yZGVyVGltZSIsImluc3RhbmNlTnVtYmVyIiwiX21heEhpc3RvcnlPcmRlclRpbWUiLCJsYXN0RGVhbFRpbWUiLCJfbWF4RGVhbFRpbWUiLCJvbkhpc3RvcnlPcmRlckFkZGVkIiwiaW5zdGFuY2VJbmRleCIsIm9uRGVhbEFkZGVkIiwiZ2V0RGVhbHNCeVRpbWVSYW5nZSIsIkRhdGUiLCJnZXREZWFsc0J5VGlja2V0IiwiaWQiLCJPYmplY3QiLCJ2YWx1ZXMiLCJfZGVhbHNCeVRpY2tldCIsInNvcnQiLCJfZGVhbHNDb21wYXJhdG9yIiwiZ2V0RGVhbHNCeVBvc2l0aW9uIiwicG9zaXRpb25JZCIsIl9kZWFsc0J5UG9zaXRpb24iLCJzdGFydFRpbWUiLCJlbmRUaW1lIiwiX2RlYWxzQnlUaW1lIiwiYmV0d2VlbkJvdW5kcyIsIiRndGUiLCJ0aW1lIiwiZW50cnlUeXBlIiwiJGx0ZSIsIk51bWJlciIsIk1BWF9WQUxVRSIsImdldEhpc3RvcnlPcmRlcnNCeVRpbWVSYW5nZSIsImdldEhpc3RvcnlPcmRlcnNCeVRpY2tldCIsIl9oaXN0b3J5T3JkZXJzQnlUaWNrZXQiLCJfaGlzdG9yeU9yZGVyc0NvbXBhcmF0b3IiLCJnZXRIaXN0b3J5T3JkZXJzQnlQb3NpdGlvbiIsIl9oaXN0b3J5T3JkZXJzQnlQb3NpdGlvbiIsIl9oaXN0b3J5T3JkZXJzQnlUaW1lIiwiZG9uZVRpbWUiLCJ0eXBlIiwic3RhdGUiLCJvbkRlYWxzU3luY2hyb25pemVkIiwic3luY2hyb25pemF0aW9uSWQiLCJfZmx1c2hEYXRhYmFzZSIsIl9vcmRlclN5bmNocm9uaXphdGlvbkZpbmlzaGVkIiwiX2RlYWxTeW5jaHJvbml6YXRpb25GaW5pc2hlZCIsIm8xIiwibzIiLCJ0aW1lRGlmZiIsImdldFRpbWUiLCJpZERpZmYiLCJBVkxUcmVlIiwiY29tcGFyZUtleXMiLCJkMSIsImQyIiwiX25ld0hpc3RvcnlPcmRlcnMiLCJfbmV3RGVhbHMiLCJjbGVhclRpbWVvdXQiLCJfZmx1c2hUaW1lb3V0IiwiZXhpc3RpbmciLCJrZXkiLCJfZ2V0RGVhbEtleSIsIm5ld0RlYWwiLCJkZWxldGUiLCJpbnNlcnQiLCJwdXNoIiwic2V0VGltZW91dCIsImJpbmQiLCJ0b0lTT1N0cmluZyIsIl9nZXRIaXN0b3J5T3JkZXJLZXkiLCJuZXdIaXN0b3J5T3JkZXIiLCJzdGF0dXMiLCJfZmx1c2hQcm9taXNlIiwiX2ZsdXNoUnVubmluZyIsInJlc29sdmUiLCJQcm9taXNlIiwicmVzIiwiZmx1c2giLCJfbG9nZ2VyIiwiZGVidWciLCJlcnIiLCJ3YXJuIiwiY29uc3RydWN0b3IiLCJIaXN0b3J5RGF0YWJhc2UiLCJnZXRJbnN0YW5jZSIsIkxvZ2dlck1hbmFnZXIiLCJnZXRMb2dnZXIiXSwibWFwcGluZ3MiOiJBQUFBOzs7Ozs7O2VBVXFCQTs7O3VFQVJNOzhEQUNDO2tDQUNKOytEQUNFOzs7Ozs7QUFLWCxJQUFBLEFBQU1BLHVCQUFOLE1BQU1BLDZCQUE2QkMsdUJBQWM7SUFZOUQ7Ozs7R0FJQyxHQUNELE1BQU1DLFdBQVdDLFNBQVMsRUFBRUMsY0FBYyxTQUFTLEVBQUU7UUFDbkQsTUFBTSxLQUFLLENBQUNGLFdBQVdDLFdBQVdDO1FBQ2xDLElBQUksRUFBQ0MsS0FBSyxFQUFFQyxhQUFhLEVBQUMsR0FBRyxNQUFNLElBQUksQ0FBQ0MsZ0JBQWdCLENBQUNDLFdBQVcsQ0FBQ0wsV0FBV0M7UUFDaEYsS0FBSyxJQUFJSyxRQUFRSixNQUFPO1lBQ3RCLE1BQU0sSUFBSSxDQUFDSyxRQUFRLENBQUNELE1BQU07UUFDNUI7UUFDQSxLQUFLLElBQUlFLGdCQUFnQkwsY0FBZTtZQUN0QyxNQUFNLElBQUksQ0FBQ00sZ0JBQWdCLENBQUNELGNBQWM7UUFDNUM7SUFDRjtJQUVBOzs7R0FHQyxHQUNELE1BQU1FLFFBQVE7UUFDWixJQUFJLENBQUNDLE1BQU07UUFDWCxNQUFNLElBQUksQ0FBQ1AsZ0JBQWdCLENBQUNNLEtBQUssQ0FBQyxJQUFJLENBQUNFLFVBQVUsRUFBRSxJQUFJLENBQUNDLFlBQVk7SUFDdEU7SUFFQTs7OztHQUlDLEdBQ0RDLHFCQUFxQkMsY0FBYyxFQUFFO1FBQ25DLE9BQU8sSUFBSSxDQUFDQyxvQkFBb0I7SUFDbEM7SUFFQTs7OztHQUlDLEdBQ0RDLGFBQWFGLGNBQWMsRUFBRTtRQUMzQixPQUFPLElBQUksQ0FBQ0csWUFBWTtJQUMxQjtJQUVBOzs7O0dBSUMsR0FDRCxNQUFNQyxvQkFBb0JDLGFBQWEsRUFBRVosWUFBWSxFQUFFO1FBQ3JELE1BQU0sSUFBSSxDQUFDQyxnQkFBZ0IsQ0FBQ0Q7SUFDOUI7SUFFQTs7OztHQUlDLEdBQ0QsTUFBTWEsWUFBWUQsYUFBYSxFQUFFZCxJQUFJLEVBQUU7UUFDckMsTUFBTSxJQUFJLENBQUNDLFFBQVEsQ0FBQ0Q7SUFDdEI7SUFFQTs7O0dBR0MsR0FDRCxJQUFJSixRQUFRO1FBQ1YsT0FBTyxJQUFJLENBQUNvQixtQkFBbUIsQ0FBQyxJQUFJQyxLQUFLLElBQUksSUFBSUEsS0FBSztJQUN4RDtJQUVBOzs7O0dBSUMsR0FDREMsaUJBQWlCQyxFQUFFLEVBQUU7UUFDbkIsSUFBSXZCLFFBQVF3QixPQUFPQyxNQUFNLENBQUMsSUFBSSxDQUFDQyxjQUFjLENBQUNILEdBQUcsSUFBSSxDQUFDO1FBQ3REdkIsTUFBTTJCLElBQUksQ0FBQyxJQUFJLENBQUNDLGdCQUFnQjtRQUNoQyxPQUFPNUI7SUFDVDtJQUVBOzs7O0dBSUMsR0FDRDZCLG1CQUFtQkMsVUFBVSxFQUFFO1FBQzdCLElBQUk5QixRQUFRd0IsT0FBT0MsTUFBTSxDQUFDLElBQUksQ0FBQ00sZ0JBQWdCLENBQUNELFdBQVcsSUFBSSxDQUFDO1FBQ2hFOUIsTUFBTTJCLElBQUksQ0FBQyxJQUFJLENBQUNDLGdCQUFnQjtRQUNoQyxPQUFPNUI7SUFDVDtJQUVBOzs7OztHQUtDLEdBQ0RvQixvQkFBb0JZLFNBQVMsRUFBRUMsT0FBTyxFQUFFO1FBQ3RDLElBQUlqQyxRQUFRLElBQUksQ0FBQ2tDLFlBQVksQ0FBQ0MsYUFBYSxDQUFDO1lBQzFDQyxNQUFNO2dCQUFDQyxNQUFNTDtnQkFBV1QsSUFBSTtnQkFBR2UsV0FBVztZQUFFO1lBQzVDQyxNQUFNO2dCQUFDRixNQUFNSjtnQkFBU1YsSUFBSWlCLE9BQU9DLFNBQVM7Z0JBQUVILFdBQVc7WUFBRTtRQUMzRDtRQUNBLE9BQU90QztJQUNUO0lBRUE7OztHQUdDLEdBQ0QsSUFBSUMsZ0JBQWdCO1FBQ2xCLE9BQU8sSUFBSSxDQUFDeUMsMkJBQTJCLENBQUMsSUFBSXJCLEtBQUssSUFBSSxJQUFJQSxLQUFLO0lBQ2hFO0lBRUE7Ozs7R0FJQyxHQUNEc0IseUJBQXlCcEIsRUFBRSxFQUFFO1FBQzNCLElBQUl0QixnQkFBZ0J1QixPQUFPQyxNQUFNLENBQUMsSUFBSSxDQUFDbUIsc0JBQXNCLENBQUNyQixHQUFHLElBQUksQ0FBQztRQUN0RXRCLGNBQWMwQixJQUFJLENBQUMsSUFBSSxDQUFDa0Isd0JBQXdCO1FBQ2hELE9BQU81QztJQUNUO0lBRUE7Ozs7R0FJQyxHQUNENkMsMkJBQTJCaEIsVUFBVSxFQUFFO1FBQ3JDLElBQUk3QixnQkFBZ0J1QixPQUFPQyxNQUFNLENBQUMsSUFBSSxDQUFDc0Isd0JBQXdCLENBQUNqQixXQUFXLElBQUksQ0FBQztRQUNoRjdCLGNBQWMwQixJQUFJLENBQUMsSUFBSSxDQUFDa0Isd0JBQXdCO1FBQ2hELE9BQU81QztJQUNUO0lBRUE7Ozs7O0dBS0MsR0FDRHlDLDRCQUE0QlYsU0FBUyxFQUFFQyxPQUFPLEVBQUU7UUFDOUMsSUFBSWhDLGdCQUFnQixJQUFJLENBQUMrQyxvQkFBb0IsQ0FBQ2IsYUFBYSxDQUFDO1lBQzFEQyxNQUFNO2dCQUFDYSxVQUFVakI7Z0JBQVdULElBQUk7Z0JBQUcyQixNQUFNO2dCQUFJQyxPQUFPO1lBQUU7WUFDdERaLE1BQU07Z0JBQUNVLFVBQVVoQjtnQkFBU1YsSUFBSWlCLE9BQU9DLFNBQVM7Z0JBQUVTLE1BQU07Z0JBQUlDLE9BQU87WUFBRTtRQUNyRTtRQUNBLE9BQU9sRDtJQUNUO0lBRUE7Ozs7OztHQU1DLEdBQ0QsTUFBTW1ELG9CQUFvQmxDLGFBQWEsRUFBRW1DLGlCQUFpQixFQUFFO1FBQzFELE1BQU0sSUFBSSxDQUFDQyxjQUFjO1FBQ3pCLE1BQU0sS0FBSyxDQUFDRixvQkFBb0JsQyxlQUFlbUM7SUFDakQ7SUFFQTVDLFNBQVM7UUFDUCxJQUFJLENBQUM4Qyw2QkFBNkIsR0FBRyxDQUFDO1FBQ3RDLElBQUksQ0FBQ0MsNEJBQTRCLEdBQUcsQ0FBQztRQUNyQyxJQUFJLENBQUM5QixjQUFjLEdBQUcsQ0FBQztRQUN2QixJQUFJLENBQUNLLGdCQUFnQixHQUFHLENBQUM7UUFDekIsSUFBSSxDQUFDYSxzQkFBc0IsR0FBRyxDQUFDO1FBQy9CLElBQUksQ0FBQ0csd0JBQXdCLEdBQUcsQ0FBQztRQUNqQyxzQ0FBc0M7UUFDdEMsSUFBSSxDQUFDRix3QkFBd0IsR0FBRyxDQUFDWSxJQUFJQztZQUNuQyxJQUFJQyxXQUFXLEFBQUNGLENBQUFBLEdBQUdSLFFBQVEsSUFBSSxJQUFJNUIsS0FBSyxFQUFDLEVBQUd1QyxPQUFPLEtBQUssQUFBQ0YsQ0FBQUEsR0FBR1QsUUFBUSxJQUFJLElBQUk1QixLQUFLLEVBQUMsRUFBR3VDLE9BQU87WUFDNUYsSUFBSUQsYUFBYSxHQUFHO2dCQUNsQixJQUFJRSxTQUFTSixHQUFHbEMsRUFBRSxHQUFHbUMsR0FBR25DLEVBQUU7Z0JBQzFCLElBQUlzQyxXQUFXLEdBQUc7b0JBQ2hCLElBQUlKLEdBQUdQLElBQUksR0FBR1EsR0FBR1IsSUFBSSxFQUFFO3dCQUNyQixPQUFPO29CQUNULE9BQU8sSUFBSU8sR0FBR1AsSUFBSSxHQUFHUSxHQUFHUixJQUFJLEVBQUU7d0JBQzVCLE9BQU8sQ0FBQztvQkFDVixPQUFPO3dCQUNMLElBQUlPLEdBQUdOLEtBQUssR0FBR08sR0FBR1AsS0FBSyxFQUFFOzRCQUN2QixPQUFPO3dCQUNULE9BQU8sSUFBSU0sR0FBR04sS0FBSyxHQUFHTyxHQUFHUCxLQUFLLEVBQUU7NEJBQzlCLE9BQU8sQ0FBQzt3QkFDVixPQUFPOzRCQUNMLE9BQU87d0JBQ1Q7b0JBQ0Y7Z0JBQ0YsT0FBTztvQkFDTCxPQUFPVTtnQkFDVDtZQUNGLE9BQU87Z0JBQ0wsT0FBT0Y7WUFDVDtRQUNGO1FBQ0EsSUFBSSxDQUFDWCxvQkFBb0IsR0FBRyxJQUFJYyx5QkFBTyxDQUFDO1lBQUNDLGFBQWEsSUFBSSxDQUFDbEIsd0JBQXdCO1FBQUE7UUFDbkYsSUFBSSxDQUFDakIsZ0JBQWdCLEdBQUcsQ0FBQ29DLElBQUlDO1lBQzNCLElBQUlOLFdBQVcsQUFBQ0ssQ0FBQUEsR0FBRzNCLElBQUksSUFBSSxJQUFJaEIsS0FBSyxFQUFDLEVBQUd1QyxPQUFPLEtBQUssQUFBQ0ssQ0FBQUEsR0FBRzVCLElBQUksSUFBSSxJQUFJaEIsS0FBSyxFQUFDLEVBQUd1QyxPQUFPO1lBQ3BGLElBQUlELGFBQWEsR0FBRztnQkFDbEIsSUFBSUUsU0FBU0csR0FBR3pDLEVBQUUsR0FBRzBDLEdBQUcxQyxFQUFFO2dCQUMxQixJQUFJc0MsV0FBVyxHQUFHO29CQUNoQixJQUFJRyxHQUFHMUIsU0FBUyxHQUFHMkIsR0FBRzNCLFNBQVMsRUFBRTt3QkFDL0IsT0FBTztvQkFDVCxPQUFPLElBQUkwQixHQUFHMUIsU0FBUyxHQUFHMkIsR0FBRzNCLFNBQVMsRUFBRTt3QkFDdEMsT0FBTyxDQUFDO29CQUNWLE9BQU87d0JBQ0wsT0FBTztvQkFDVDtnQkFDRixPQUFPO29CQUNMLE9BQU91QjtnQkFDVDtZQUNGLE9BQU87Z0JBQ0wsT0FBT0Y7WUFDVDtRQUNGO1FBQ0EsSUFBSSxDQUFDekIsWUFBWSxHQUFHLElBQUk0Qix5QkFBTyxDQUFDO1lBQUNDLGFBQWEsSUFBSSxDQUFDbkMsZ0JBQWdCO1FBQUE7UUFDbkUsSUFBSSxDQUFDZCxvQkFBb0IsR0FBRyxJQUFJTyxLQUFLO1FBQ3JDLElBQUksQ0FBQ0wsWUFBWSxHQUFHLElBQUlLLEtBQUs7UUFDN0IsSUFBSSxDQUFDNkMsaUJBQWlCLEdBQUcsRUFBRTtRQUMzQixJQUFJLENBQUNDLFNBQVMsR0FBRyxFQUFFO1FBQ25CQyxhQUFhLElBQUksQ0FBQ0MsYUFBYTtRQUMvQixPQUFPLElBQUksQ0FBQ0EsYUFBYTtJQUMzQjtJQUVBLHNDQUFzQztJQUN0QyxNQUFNaEUsU0FBU0QsSUFBSSxFQUFFa0UsUUFBUSxFQUFFO1FBQzdCLElBQUlDLE1BQU0sSUFBSSxDQUFDQyxXQUFXLENBQUNwRTtRQUMzQixJQUFJLENBQUNzQixjQUFjLENBQUN0QixLQUFLbUIsRUFBRSxDQUFDLEdBQUcsSUFBSSxDQUFDRyxjQUFjLENBQUN0QixLQUFLbUIsRUFBRSxDQUFDLElBQUksQ0FBQztRQUNoRSxJQUFJa0QsVUFBVSxDQUFDSCxZQUFZLENBQUMsSUFBSSxDQUFDNUMsY0FBYyxDQUFDdEIsS0FBS21CLEVBQUUsQ0FBQyxDQUFDZ0QsSUFBSTtRQUM3RCxJQUFJLENBQUM3QyxjQUFjLENBQUN0QixLQUFLbUIsRUFBRSxDQUFDLENBQUNnRCxJQUFJLEdBQUduRTtRQUNwQyxJQUFJQSxLQUFLMEIsVUFBVSxFQUFFO1lBQ25CLElBQUksQ0FBQ0MsZ0JBQWdCLENBQUMzQixLQUFLMEIsVUFBVSxDQUFDLEdBQUcsSUFBSSxDQUFDQyxnQkFBZ0IsQ0FBQzNCLEtBQUswQixVQUFVLENBQUMsSUFBSSxDQUFDO1lBQ3BGLElBQUksQ0FBQ0MsZ0JBQWdCLENBQUMzQixLQUFLMEIsVUFBVSxDQUFDLENBQUN5QyxJQUFJLEdBQUduRTtRQUNoRDtRQUNBLElBQUksQ0FBQzhCLFlBQVksQ0FBQ3dDLE1BQU0sQ0FBQ3RFO1FBQ3pCLElBQUksQ0FBQzhCLFlBQVksQ0FBQ3lDLE1BQU0sQ0FBQ3ZFLE1BQU1BO1FBQy9CLElBQUlBLEtBQUtpQyxJQUFJLElBQUssQ0FBQSxDQUFDLElBQUksQ0FBQ3JCLFlBQVksSUFBSSxJQUFJLENBQUNBLFlBQVksQ0FBQzRDLE9BQU8sS0FBS3hELEtBQUtpQyxJQUFJLENBQUN1QixPQUFPLEVBQUMsR0FBSTtZQUMxRixJQUFJLENBQUM1QyxZQUFZLEdBQUdaLEtBQUtpQyxJQUFJO1FBQy9CO1FBQ0EsSUFBSW9DLFNBQVM7WUFDWCxJQUFJLENBQUNOLFNBQVMsQ0FBQ1MsSUFBSSxDQUFDeEU7WUFDcEJnRSxhQUFhLElBQUksQ0FBQ0MsYUFBYTtZQUMvQixJQUFJLENBQUNBLGFBQWEsR0FBR1EsV0FBVyxJQUFJLENBQUN2QixjQUFjLENBQUN3QixJQUFJLENBQUMsSUFBSSxHQUFHO1FBQ2xFO0lBQ0Y7SUFFQU4sWUFBWXBFLElBQUksRUFBRTtRQUNoQixPQUFPLEFBQUNBLENBQUFBLEtBQUtpQyxJQUFJLElBQUksSUFBSWhCLEtBQUssRUFBQyxFQUFHMEQsV0FBVyxLQUFLLE1BQU0zRSxLQUFLbUIsRUFBRSxHQUFHLE1BQU1uQixLQUFLa0MsU0FBUztJQUN4RjtJQUVBLHNDQUFzQztJQUN0QyxNQUFNL0IsaUJBQWlCRCxZQUFZLEVBQUVnRSxRQUFRLEVBQUU7UUFDN0MsSUFBSUMsTUFBTSxJQUFJLENBQUNTLG1CQUFtQixDQUFDMUU7UUFDbkMsSUFBSSxDQUFDc0Msc0JBQXNCLENBQUN0QyxhQUFhaUIsRUFBRSxDQUFDLEdBQUcsSUFBSSxDQUFDcUIsc0JBQXNCLENBQUN0QyxhQUFhaUIsRUFBRSxDQUFDLElBQUksQ0FBQztRQUNoRyxJQUFJMEQsa0JBQWtCLENBQUNYLFlBQVksQ0FBQyxJQUFJLENBQUMxQixzQkFBc0IsQ0FBQ3RDLGFBQWFpQixFQUFFLENBQUMsQ0FBQ2dELElBQUk7UUFDckYsSUFBSSxDQUFDM0Isc0JBQXNCLENBQUN0QyxhQUFhaUIsRUFBRSxDQUFDLENBQUNnRCxJQUFJLEdBQUdqRTtRQUNwRCxJQUFJQSxhQUFhd0IsVUFBVSxFQUFFO1lBQzNCLElBQUksQ0FBQ2lCLHdCQUF3QixDQUFDekMsYUFBYXdCLFVBQVUsQ0FBQyxHQUFHLElBQUksQ0FBQ2lCLHdCQUF3QixDQUFDekMsYUFBYXdCLFVBQVUsQ0FBQyxJQUM3RyxDQUFDO1lBQ0gsSUFBSSxDQUFDaUIsd0JBQXdCLENBQUN6QyxhQUFhd0IsVUFBVSxDQUFDLENBQUN5QyxJQUFJLEdBQUdqRTtRQUNoRTtRQUNBLElBQUksQ0FBQzBDLG9CQUFvQixDQUFDMEIsTUFBTSxDQUFDcEU7UUFDakMsSUFBSSxDQUFDMEMsb0JBQW9CLENBQUMyQixNQUFNLENBQUNyRSxjQUFjQTtRQUMvQyxJQUFJQSxhQUFhMkMsUUFBUSxJQUFLLENBQUEsQ0FBQyxJQUFJLENBQUNuQyxvQkFBb0IsSUFDcEQsSUFBSSxDQUFDQSxvQkFBb0IsQ0FBQzhDLE9BQU8sS0FBS3RELGFBQWEyQyxRQUFRLENBQUNXLE9BQU8sRUFBQyxHQUFJO1lBQzFFLElBQUksQ0FBQzlDLG9CQUFvQixHQUFHUixhQUFhMkMsUUFBUTtRQUNuRDtRQUNBLElBQUlnQyxpQkFBaUI7WUFDbkIsSUFBSSxDQUFDZixpQkFBaUIsQ0FBQ1UsSUFBSSxDQUFDdEU7WUFDNUI4RCxhQUFhLElBQUksQ0FBQ0MsYUFBYTtZQUMvQixJQUFJLENBQUNBLGFBQWEsR0FBR1EsV0FBVyxJQUFJLENBQUN2QixjQUFjLENBQUN3QixJQUFJLENBQUMsSUFBSSxHQUFHO1FBQ2xFO0lBQ0Y7SUFFQUUsb0JBQW9CMUUsWUFBWSxFQUFFO1FBQ2hDLE9BQU8sQUFBQ0EsQ0FBQUEsYUFBYTJDLFFBQVEsSUFBSSxJQUFJNUIsS0FBSyxFQUFDLEVBQUcwRCxXQUFXLEtBQUssTUFBTXpFLGFBQWFpQixFQUFFLEdBQUcsTUFDcEZqQixhQUFhNEMsSUFBSSxHQUFHLE1BQU01QyxhQUFhNEUsTUFBTTtJQUNqRDtJQUVBLE1BQU01QixpQkFBaUI7UUFDckIsSUFBSSxJQUFJLENBQUM2QixhQUFhLEVBQUU7WUFDdEIsTUFBTSxJQUFJLENBQUNBLGFBQWE7UUFDMUI7UUFDQSxJQUFJLElBQUksQ0FBQ0MsYUFBYSxFQUFFO1lBQ3RCO1FBQ0Y7UUFDQSxJQUFJLENBQUNBLGFBQWEsR0FBRztRQUNyQixJQUFJQztRQUNKLElBQUksQ0FBQ0YsYUFBYSxHQUFHLElBQUlHLFFBQVFDLENBQUFBLE1BQU9GLFVBQVVFO1FBQ2xELElBQUk7WUFDRixNQUFNLElBQUksQ0FBQ3JGLGdCQUFnQixDQUFDc0YsS0FBSyxDQUFDLElBQUksQ0FBQzlFLFVBQVUsRUFBRSxJQUFJLENBQUNDLFlBQVksRUFBRSxJQUFJLENBQUN1RCxpQkFBaUIsRUFBRSxJQUFJLENBQUNDLFNBQVM7WUFDNUcsSUFBSSxDQUFDRCxpQkFBaUIsR0FBRyxFQUFFO1lBQzNCLElBQUksQ0FBQ0MsU0FBUyxHQUFHLEVBQUU7WUFDbkIsSUFBSSxDQUFDc0IsT0FBTyxDQUFDQyxLQUFLLENBQUMsQ0FBQyxFQUFFLElBQUksQ0FBQ2hGLFVBQVUsQ0FBQyxvQkFBb0IsQ0FBQztRQUM3RCxFQUFFLE9BQU9pRixLQUFLO1lBQ1osSUFBSSxDQUFDRixPQUFPLENBQUNHLElBQUksQ0FBQyxDQUFDLEVBQUUsSUFBSSxDQUFDbEYsVUFBVSxDQUFDLDJCQUEyQixDQUFDLEVBQUVpRjtZQUNuRSxJQUFJLENBQUN0QixhQUFhLEdBQUdRLFdBQVcsSUFBSSxDQUFDdkIsY0FBYyxDQUFDd0IsSUFBSSxDQUFDLElBQUksR0FBRztRQUNsRSxTQUFVO1lBQ1JPO1lBQ0EsSUFBSSxDQUFDRCxhQUFhLEdBQUc7UUFDdkI7SUFDRjtJQXRUQTs7R0FFQyxHQUNEUyxhQUFjO1FBQ1osS0FBSztRQUNMLElBQUksQ0FBQzNGLGdCQUFnQixHQUFHNEYsY0FBZSxDQUFDQyxXQUFXO1FBQ25ELElBQUksQ0FBQ3RGLE1BQU07UUFDWCxJQUFJLENBQUNnRixPQUFPLEdBQUdPLGVBQWEsQ0FBQ0MsU0FBUyxDQUFDO0lBQ3pDO0FBZ1RGIn0=