"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "default", {
    enumerable: true,
    get: function() {
        return MetatraderAccountApi;
    }
});
const _metatraderAccount = /*#__PURE__*/ _interop_require_default(require("./metatraderAccount"));
const _metatraderAccountReplica = /*#__PURE__*/ _interop_require_default(require("./metatraderAccountReplica"));
const _metatraderAccountApi = require("./metatraderAccountApi");
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
let MetatraderAccountApi = class MetatraderAccountApi {
    /**
   * Returns trading accounts belonging to the current user, provides pagination in infinite scroll style
   * @param {AccountsFilter} [accountsFilter] optional filter
   * @return {Promise<Array<MetatraderAccount>>} promise resolving with MetaTrader account entities
   */ async getAccountsWithInfiniteScrollPagination(accountsFilter) {
        let accounts = await this._metatraderAccountClient.getAccounts(accountsFilter, "1");
        return accounts.map((a)=>new _metatraderAccount.default(a, this._metatraderAccountClient, this._metaApiWebsocketClient, this._connectionRegistry, this._expertAdvisorClient, this._historicalMarketDataClient, this._application));
    }
    /**
   * Returns trading accounts belonging to the current user with accounts count,
   * provides pagination in a classic style
   * @param {AccountsFilter} [accountsFilter] optional filter
   * @return {Promise<MetatraderAccountList>} promise resolving with an array of MetaTrader account entities and count
   */ async getAccountsWithClassicPagination(accountsFilter) {
        let accounts = await this._metatraderAccountClient.getAccounts(accountsFilter, "2");
        return {
            count: accounts.count,
            items: accounts.items.map((a)=>new _metatraderAccount.default(a, this._metatraderAccountClient, this._metaApiWebsocketClient, this._connectionRegistry, this._expertAdvisorClient, this._historicalMarketDataClient, this._application))
        };
    }
    /**
   * Returns trading account by id
   * @param {string} accountId MetaTrader account id
   * @return {Promise<MetatraderAccount>} promise resolving with MetaTrader account entity
   */ async getAccount(accountId) {
        let account = await this._metatraderAccountClient.getAccount(accountId);
        return new _metatraderAccount.default(account, this._metatraderAccountClient, this._metaApiWebsocketClient, this._connectionRegistry, this._expertAdvisorClient, this._historicalMarketDataClient, this._application);
    }
    /**
   * Returns trading account replica by trading account id and replica id
   * @param {string} accountId MetaTrader primary account id
   * @param {string} replicaId MetaTrader account replica id
   * @return {Promise<MetatraderAccountReplica>} promise resolving with MetaTrader account replica found
   */ async getAccountReplica(accountId, replicaId) {
        let account = await this._metatraderAccountClient.getAccount(accountId);
        let replica = await this._metatraderAccountClient.getAccountReplica(accountId, replicaId);
        return new _metatraderAccountReplica.default(replica, account, this._metatraderAccountClient);
    }
    /**
   * Returns replicas for a trading account
   * @param {string} accountId Primary account id
   * @return {Promise<Array<MetatraderAccountReplica>>} promise resolving with MetaTrader account replicas found
   */ async getAccountReplicas(accountId) {
        let account = await this._metatraderAccountClient.getAccount(accountId);
        let replicas = await this._metatraderAccountClient.getAccountReplicas(accountId);
        if (replicas.items) {
            replicas = replicas.items;
        }
        return replicas.map((replica)=>new _metatraderAccountReplica.default(replica, account, this._metatraderAccountClient));
    }
    /**
   * Adds a trading account and starts a cloud API server for the trading account
   * @param {NewMetatraderAccountDto} account MetaTrader account data
   * @return {Promise<MetatraderAccount>} promise resolving with created MetaTrader account entity
   */ async createAccount(account) {
        let id = await this._metatraderAccountClient.createAccount(account);
        return this.getAccount(id.id);
    }
    /**
   * Constructs a MetaTrader account API instance
   * @param {MetatraderAccountClient} metatraderAccountClient MetaTrader account REST API client
   * @param {MetaApiWebsocketClient} metaApiWebsocketClient MetaApi websocket client
   * @param {ConnectionRegistry} connectionRegistry metatrader account connection registry
   * @param {ExpertAdvisorClient} expertAdvisorClient expert advisor REST API client
   * @param {HistoricalMarketDataClient} historicalMarketDataClient historical market data HTTP API client
   * @param {string} application application name
   */ constructor(metatraderAccountClient, metaApiWebsocketClient, connectionRegistry, expertAdvisorClient, historicalMarketDataClient, application){
        this._metatraderAccountClient = metatraderAccountClient;
        this._metaApiWebsocketClient = metaApiWebsocketClient;
        this._connectionRegistry = connectionRegistry;
        this._expertAdvisorClient = expertAdvisorClient;
        this._historicalMarketDataClient = historicalMarketDataClient;
        this._application = application;
    }
};

//# sourceMappingURL=data:application/json;base64,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