import MetaApiWebsocketClient, {
  GetAccountInformationOptions, GetOrderOptions, GetOrdersOptions, GetPositionOptions, GetPositionsOptions,
  MetatraderAccountInformation, MetatraderCandle, MetatraderDeals, MetatraderHistoryOrders, MetatraderOrder,
  MetatraderPosition, MetatraderSymbolPrice, MetatraderSymbolSpecification, MetatraderTick, ServerTime
} from '../clients/metaApi/metaApiWebsocket.client';
import MetaApiConnectionInstance from './metaApiConnectionInstance';
import StreamingMetaApiConnection from './streamingMetaApiConnection';

/**
 * Exposes MetaApi MetaTrader RPC API connection instance to consumers
 */
export default class RpcMetaApiConnectionInstance extends MetaApiConnectionInstance {
  
  /**
   * Constructs MetaApi MetaTrader RPC Api connection instance
   * @param {MetaApiWebsocketClient} websocketClient MetaApi websocket client
   * @param {StreamingMetaApiConnection} metaApiConnection RPC MetaApi connection
   */
  constructor(websocketClient: MetaApiWebsocketClient, metaApiConnection: StreamingMetaApiConnection);
  
  /**
   * Returns account information
   * @param {GetAccountInformationOptions} [options] additional request options
   * @returns {Promise<MetatraderAccountInformation>} promise resolving with account information
   */
  getAccountInformation(options?: GetAccountInformationOptions): Promise<MetatraderAccountInformation>;
  
  /**
   * Returns positions
   * @param {GetPositionsOptions} [options] additional request options
   * @returns {Promise<Array<MetatraderPosition>} promise resolving with array of open positions
   */
  getPositions(options?: GetPositionsOptions): Promise<Array<MetatraderPosition>>;
  
  /**
   * Returns specific position
   * @param {string} positionId position id
   * @param {GetPositionOptions} [options] additional request options
   * @return {Promise<MetatraderPosition>} promise resolving with MetaTrader position found
   */
  getPosition(positionId: string, options?: GetPositionOptions): Promise<MetatraderPosition>;
  
  /**
   * Returns open orders
   * @param {GetOrdersOptions} [options] additional request options
   * @return {Promise<Array<MetatraderOrder>>} promise resolving with open MetaTrader orders
   */
  getOrders(options?: GetOrdersOptions): Promise<Array<MetatraderOrder>>;
  
  /**
   * Returns specific open order
   * @param {string} orderId order id (ticket number)
   * @param {GetOrderOptions} [options] additional request options
   * @return {Promise<MetatraderOrder>} promise resolving with metatrader order found
   */
  getOrder(orderId: string, options?: GetOrderOptions): Promise<MetatraderOrder>;
  
  /**
   * Returns the history of completed orders for a specific ticket number
   * @param {string} ticket ticket number (order id)
   * @returns {Promise<MetatraderHistoryOrders>} promise resolving with request results containing history orders found
   */
  getHistoryOrdersByTicket(ticket: string): Promise<MetatraderHistoryOrders>;
  
  /**
   * Returns the history of completed orders for a specific position id
   * @param {string} positionId position id
   * @returns {Promise<MetatraderHistoryOrders>} promise resolving with request results containing history orders found
   */
  getHistoryOrdersByPosition(positionId: string): Promise<MetatraderHistoryOrders>;
  
  /**
   * Returns the history of completed orders for a specific time range
   * @param {Date} startTime start of time range, inclusive
   * @param {Date} endTime end of time range, exclusive
   * @param {number} offset pagination offset, default is 0
   * @param {number} limit pagination limit, default is 1000
   * @returns {Promise<MetatraderHistoryOrders>} promise resolving with request results containing history orders found
   */
  getHistoryOrdersByTimeRange(startTime: Date, endTime: Date, offset?: number, limit?: number): Promise<MetatraderHistoryOrders>;
  
  /**
   * Returns history deals with a specific ticket number
   * @param {string} ticket ticket number (deal id for MT5 or order id for MT4)
   * @returns {Promise<MetatraderDeals>} promise resolving with request results containing deals found
   */
  getDealsByTicket(ticket: string): Promise<MetatraderDeals>;
  
  /**
   * Returns history deals for a specific position id
   * @param {string} positionId position id
   * @returns {Promise<MetatraderDeals>} promise resolving with request results containing deals found
   */
  getDealsByPosition(positionId: string): Promise<MetatraderDeals>;
  
  /**
   * Returns history deals with for a specific time range
   * @param {Date} startTime start of time range, inclusive
   * @param {Date} endTime end of time range, exclusive
   * @param {number} offset pagination offset, default is 0
   * @param {number} limit pagination limit, default is 1000
   * @returns {Promise<MetatraderDeals>} promise resolving with request results containing deals found
   */
  getDealsByTimeRange(startTime: Date, endTime: Date, offset?: number, limit?: number): Promise<MetatraderDeals>;
  
  /**
   * Retrieves available symbols for an account
   * @returns {Promise<Array<string>>} promise which resolves when symbols are retrieved
   */
  getSymbols(): Promise<Array<string>>;
  
  /**
   * Retrieves specification for a symbol
   * @param {string} symbol symbol to retrieve specification for
   * @returns {Promise<MetatraderSymbolSpecification>} promise which resolves when specification is retrieved
   */
  getSymbolSpecification(symbol: string): Promise<MetatraderSymbolSpecification>;
  
  /**
   * Retrieves latest price for a symbol
   * @param {string} symbol symbol to retrieve price for
   * @param {boolean} keepSubscription if set to true, the account will get a long-term subscription to symbol market
   * data. Long-term subscription means that on subsequent calls you will get updated value faster. If set to false or
   * not set, the subscription will be set to expire in 12 minutes.
   * @returns {Promise<MetatraderSymbolPrice>} promise which resolves when price is retrieved
   */
  getSymbolPrice(symbol: string, keepSubscription: boolean): Promise<MetatraderSymbolPrice>;
  
  /**
   * Retrieves latest candle for a symbol and timeframe
   * @param {string} symbol symbol to retrieve candle for
   * @param {string} timeframe defines the timeframe according to which the candle must be generated. Allowed values for
   * MT5 are 1m, 2m, 3m, 4m, 5m, 6m, 10m, 12m, 15m, 20m, 30m, 1h, 2h, 3h, 4h, 6h, 8h, 12h, 1d, 1w, 1mn. Allowed values
   * for MT4 are 1m, 5m, 15m 30m, 1h, 4h, 1d, 1w, 1mn
   * @param {boolean} keepSubscription if set to true, the account will get a long-term subscription to symbol market
   * data. Long-term subscription means that on subsequent calls you will get updated value faster. If set to false or
   * not set, the subscription will be set to expire in 12 minutes.
   * @returns {Promise<MetatraderCandle>} promise which resolves when candle is retrieved
   */
  getCandle(symbol: string, timeframe: string, keepSubscription?: boolean): Promise<MetatraderCandle>;
  
  /**
   * Retrieves latest tick for a symbol. MT4 G1 accounts do not support this API
   * @param {string} symbol symbol to retrieve tick for
   * @param {boolean} keepSubscription if set to true, the account will get a long-term subscription to symbol market
   * data. Long-term subscription means that on subsequent calls you will get updated value faster. If set to false or
   * not set, the subscription will be set to expire in 12 minutes.
   * @returns {Promise<MetatraderTick>} promise which resolves when tick is retrieved
   */
  getTick(symbol: string, keepSubscription?: boolean): Promise<MetatraderTick>;
  
  /**
   * Retrieves latest order book for a symbol. MT4 accounts do not support this API
   * @param {string} symbol symbol to retrieve order book for
   * @param {boolean} keepSubscription if set to true, the account will get a long-term subscription to symbol market
   * data. Long-term subscription means that on subsequent calls you will get updated value faster. If set to false or
   * not set, the subscription will be set to expire in 12 minutes.
   * @returns {Promise<MetatraderTick>} promise which resolves when order book is retrieved
   */
  getBook(symbol: string, keepSubscription?: boolean): Promise<MetatraderTick>;

  /**
   * Returns server time for a specified MetaTrader account
   * @returns {Promise<ServerTime>} promise resolving with server time
   */
  getServerTime(): Promise<ServerTime>;

  /**
   * Waits until synchronization to RPC application is completed
   * @param {number} timeoutInSeconds synchronization timeout in seconds
   * @return {Promise} promise which resolves when synchronization to RPC application is completed
   */
  waitSynchronized(timeoutInSeconds?: number): Promise<any>;
}