'use strict';

import MetaApiWebsocketClient from '../clients/metaApi/metaApiWebsocket.client';
import MetaApiConnection from './metaApiConnection';
import MetatraderAccount from './metatraderAccount';
import * as helpers from '../helpers/helpers';
import sinon from 'sinon';

/**
 * @test {MetaApiConnection}
 */
describe('MetaApiConnection', () => {

  let options = {
    region: undefined,
    connections: {
      refreshReplicasMaxDelayInMs: 0
    }
  };

  let sandbox = sinon.createSandbox();
  let websocketClient;
  let connection;
  let account;

  beforeEach(() => {
    websocketClient = sandbox.createStubInstance(MetaApiWebsocketClient);
    account = sandbox.createStubInstance(MetatraderAccount);
    connection = new MetaApiConnection(options, websocketClient, account);
  });

  afterEach(() => {
    sandbox.restore();
  });

  /**
   * @test {MetaApiConnection#scheduleRefresh}
   */
  describe('scheduleRefresh', () => {

    /**
     * @test {MetaApiConnection#scheduleRefresh}
     */
    it('should subscribe to replicas of all regions if the region is not filtered out explicitly', async () => {
      let replicas = [
        {id: 'replica1', region: 'region2'}
      ];
      let accountRegions = {
        'region1': 'account1',
        'region2': 'replica1'
      };
      sandbox.stub(account, 'id').get(() => 'account1');
      sandbox.stub(account, 'region').get(() => 'region');
      sandbox.stub(account, 'replicas').get(() => replicas);
      sandbox.stub(account, 'accountRegions').get(() => accountRegions);
      account.reload.callsFake(async () => {
        replicas.push({id: 'replica2', region: 'region3'});
        accountRegions.region3 = 'replica2';
      });

      connection.scheduleRefresh('region');
      await helpers.delay(25);
      
      sinon.assert.callCount(websocketClient.ensureSubscribe, 6);
      sinon.assert.calledWith(websocketClient.ensureSubscribe, 'account1', 0);
      sinon.assert.calledWith(websocketClient.ensureSubscribe, 'account1', 1);
      sinon.assert.calledWith(websocketClient.ensureSubscribe, 'replica1', 0);
      sinon.assert.calledWith(websocketClient.ensureSubscribe, 'replica1', 1);
      sinon.assert.calledWith(websocketClient.ensureSubscribe, 'replica2', 0);
      sinon.assert.calledWith(websocketClient.ensureSubscribe, 'replica2', 1);
    });

    /**
     * @test {MetaApiConnection#scheduleRefresh}
     */
    it('should subscribe to replica of only explicitly configured region when replicas change', async () => {
      sandbox.stub(options, 'region').value('region2');
      
      let replicas = [
        {id: 'replica1', region: 'region2'}
      ];
      let accountRegions = {
        'region1': 'account1',
        'region2': 'replica1'
      };
      sandbox.stub(account, 'id').get(() => 'account1');
      sandbox.stub(account, 'region').get(() => 'region');
      sandbox.stub(account, 'replicas').get(() => replicas);
      sandbox.stub(account, 'accountRegions').get(() => accountRegions);
      account.reload.callsFake(async () => {
        replicas.push({id: 'replica2', region: 'region3'});
        accountRegions.region3 = 'replica2';
      });

      connection.scheduleRefresh('region');
      await helpers.delay(25);

      sinon.assert.callCount(websocketClient.ensureSubscribe, 2);
      sinon.assert.calledWith(websocketClient.ensureSubscribe, 'replica1', 0);
      sinon.assert.calledWith(websocketClient.ensureSubscribe, 'replica1', 1);
    });

  });

});
